<?php
session_start();
require 'db.php';
require 'lang.php';

if (!isset($_SESSION['driver_id'])) { 
    header("Location: driver_login.php"); 
    exit; 
}
$driver_id = $_SESSION['driver_id'];

if (isset($_GET['ajax'])) {
    header('Content-Type: application/json');
    
    if ($_GET['ajax'] == 'take_order' && isset($_GET['order_id'])) {
        $order_id = (int)$_GET['order_id'];
        $stmt = $pdo->prepare("UPDATE orders SET driver_id=?, status='accepted' WHERE id=? AND status='pending'");
        $success = $stmt->execute([$driver_id, $order_id]);
        
        if ($success) {
            echo json_encode(['success' => true]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Order no longer available']);
        }
        exit;
    }
    
    if ($_GET['ajax'] == 'mark_delivered' && isset($_GET['order_id'])) {
        $order_id = (int)$_GET['order_id'];
        
        $pdo->beginTransaction();
        try {
            $stmt = $pdo->prepare("UPDATE orders SET status='delivered' WHERE id=? AND driver_id=?");
            $stmt->execute([$order_id, $driver_id]);
            
            $stmt2 = $pdo->prepare("UPDATE drivers SET empty_stock = IFNULL(empty_stock,0) + 1 WHERE id=?");
            $stmt2->execute([$driver_id]);
            
            $pdo->commit();
            echo json_encode(['success' => true]);
        } catch (Exception $e) {
            $pdo->rollBack();
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
        exit;
    }
    
    if ($_GET['ajax'] == 'update_stock') {
        $manual_sell = (int)($_POST['manual_sell'] ?? 0);
        $exchange = (int)($_POST['exchange'] ?? 0);

        if ($manual_sell > 0) {
            $stmt = $pdo->prepare("UPDATE drivers 
                                   SET stock = GREATEST(IFNULL(stock,0) - ?,0),
                                       sold = IFNULL(sold,0) + ?
                                   WHERE id=?");
            $stmt->execute([$manual_sell, $manual_sell, $driver_id]);
        }

        if ($exchange > 0) {
            $stmt = $pdo->prepare("UPDATE drivers SET empty_stock = IFNULL(empty_stock,0) + ? WHERE id=?");
            $stmt->execute([$exchange, $driver_id]);
        }

        $stmt = $pdo->prepare("UPDATE drivers 
                               SET filled_stock = GREATEST(IFNULL(stock,0) - IFNULL(sold,0) - IFNULL(empty_stock,0),0)
                               WHERE id=?");
        $stmt->execute([$driver_id]);
        
        echo json_encode(['success' => true]);
        exit;
    }
    
    if ($_GET['ajax'] == 'get_updates') {
        $stmt = $pdo->prepare("SELECT * FROM drivers WHERE id=?");
        $stmt->execute([$driver_id]);
        $driver = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $ordersStmt = $pdo->prepare("SELECT o.*, u.name AS user_name, u.phone AS user_phone, u.address, u.gps_lat, u.gps_lng, u.profile_pic
                                     FROM orders o 
                                     JOIN users u ON o.user_id=u.id 
                                     WHERE o.area_id=? 
                                     ORDER BY o.created_at DESC");
        $ordersStmt->execute([$driver['area_id']]);
        $orders = $ordersStmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'driver' => $driver,
            'orders' => $orders
        ]);
        exit;
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && !isset($_GET['ajax'])) {
    $manual_sell = (int)($_POST['manual_sell'] ?? 0);
    $exchange = (int)($_POST['exchange'] ?? 0);

    if ($manual_sell > 0) {
        $stmt = $pdo->prepare("UPDATE drivers 
                               SET stock = GREATEST(IFNULL(stock,0) - ?,0),
                                   sold = IFNULL(sold,0) + ?
                               WHERE id=?");
        $stmt->execute([$manual_sell, $manual_sell, $driver_id]);
    }

    if ($exchange > 0) {
        $stmt = $pdo->prepare("UPDATE drivers SET empty_stock = IFNULL(empty_stock,0) + ? WHERE id=?");
        $stmt->execute([$exchange, $driver_id]);
    }

    $stmt = $pdo->prepare("UPDATE drivers 
                           SET filled_stock = GREATEST(IFNULL(stock,0) - IFNULL(sold,0) - IFNULL(empty_stock,0),0)
                           WHERE id=?");
        $stmt->execute([$driver_id]);

    header("Location: driver_panel.php");
    exit;
}

if (isset($_GET['take']) && !isset($_GET['ajax'])) {
    $stmt = $pdo->prepare("UPDATE orders SET driver_id=?, status='accepted' WHERE id=? AND status='pending'");
    $stmt->execute([$driver_id, (int)$_GET['take']]);
    header("Location: driver_panel.php");
    exit;
}

if (isset($_GET['delivered']) && !isset($_GET['ajax'])) {
    $orderId = (int)$_GET['delivered'];
    $stmtCheck = $pdo->prepare("SELECT status FROM orders WHERE id=? AND driver_id=?");
    $stmtCheck->execute([$orderId, $driver_id]);
    $orderStatus = $stmtCheck->fetchColumn();
    if ($orderStatus === 'accepted') {
        $pdo->beginTransaction();
        try {
            $stmt = $pdo->prepare("UPDATE orders SET status='delivered' WHERE id=? AND driver_id=?");
            $stmt->execute([$orderId, $driver_id]);
            $stmt2 = $pdo->prepare("UPDATE drivers SET empty_stock = IFNULL(empty_stock,0) + 1 WHERE id=?");
            $stmt2->execute([$driver_id]);
            $pdo->commit();
            header("Location: driver_panel.php");
            exit;
        } catch (Exception $e) {
            $pdo->rollBack();
            echo "Error: " . $e->getMessage();
        }
    } else {
        header("Location: driver_panel.php");
        exit;
    }
}

$stmt = $pdo->prepare("SELECT * FROM drivers WHERE id=?");
$stmt->execute([$driver_id]);
$driver = $stmt->fetch(PDO::FETCH_ASSOC);

$ordersStmt = $pdo->prepare("SELECT o.*, u.name AS user_name, u.phone AS user_phone, u.address, u.gps_lat, u.gps_lng, u.profile_pic
                             FROM orders o 
                             JOIN users u ON o.user_id=u.id 
                             WHERE o.area_id=? 
                             ORDER BY o.created_at DESC");
$ordersStmt->execute([$driver['area_id']]);
$orders = $ordersStmt->fetchAll(PDO::FETCH_ASSOC);

$pendingOrders = array_filter($orders, function($order) {
    return $order['status'] === 'pending';
});
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title><?php echo t('driver_panel'); ?> - <?php echo htmlspecialchars($driver['name']); ?></title>
  <link rel="stylesheet" href="assets/css/styles.css">
  <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
  <style>
    .profile-pic { width:40px; height:40px; border-radius:50%; object-fit:cover; margin-right:5px; vertical-align:middle; }
    .badge.pending { background:#f0ad4e; color:#fff; padding:2px 6px; border-radius:3px; }
    .badge.accepted { background:#5bc0de; color:#fff; padding:2px 6px; border-radius:3px; }
    .badge.delivered { background:#5cb85c; color:#fff; padding:2px 6px; border-radius:3px; }
    .notification {
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px;
        background: #5cb85c;
        color: white;
        border-radius: 5px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.2);
        z-index: 1000;
        display: none;
    }
    .connection-status {
        display: inline-block;
        padding: 4px 8px;
        border-radius: 4px;
        color: white;
        font-size: 12px;
        margin-right: 10px;
    }
    .connection-status.connected { background: #5cb85c; }
    .connection-status.disconnected { background: #d9534f; }
    .pulse {
        animation: pulse 2s infinite;
    }
    @keyframes pulse {
        0% { opacity: 1; }
        50% { opacity: 0.5; }
        100% { opacity: 1; }
    }
    .language-selector {
        display: inline-block;
        margin-right: 15px;
    }
    .language-selector select {
        padding: 5px;
        border-radius: 3px;
        border: 1px solid #ddd;
    }
    
    /* Orders Card Layout */
    .orders-grid {
        display: grid;
        grid-template-columns: repeat(3, 1fr);
        gap: 20px;
        margin-top: 20px;
    }
    
    .order-card {
        background: #2c3e50;
        border: 1px solid #34495e;
        border-radius: 10px;
        padding: 20px;
        transition: all 0.3s ease;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }
    
    .order-card:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 15px rgba(0,0,0,0.2);
        border-color: #3498db;
    }
    
    .order-card-header {
        display: flex;
        align-items: center;
        margin-bottom: 15px;
        border-bottom: 1px solid #34495e;
        padding-bottom: 10px;
    }
    
    .order-card-user {
        flex: 1;
    }
    
    .order-card-user-name {
        font-weight: 600;
        color: #ecf0f1;
        margin-bottom: 5px;
    }
    
    .order-card-user-phone {
        color: #bdc3c7;
        font-size: 0.9rem;
    }
    
    .order-card-badge {
        font-size: 0.8rem;
        padding: 4px 8px;
        border-radius: 4px;
    }
    
    .order-card-details {
        margin-bottom: 15px;
    }
    
    .order-card-detail {
        display: flex;
        align-items: center;
        margin-bottom: 8px;
        font-size: 0.9rem;
    }
    
    .order-card-detail i {
        margin-right: 8px;
        color: #3498db;
        width: 16px;
        text-align: center;
    }
    
    .order-card-address {
        color: #bdc3c7;
        line-height: 1.4;
    }
    
    .order-card-actions {
        display: flex;
        gap: 10px;
        margin-top: 15px;
    }
    
    .order-card-action {
        flex: 1;
        text-align: center;
        padding: 8px 12px;
        font-size: 0.85rem;
    }
    
    /* Hide the old table */
    #orders-table-container {
        display: none;
    }
    
    /* Map Modal Styles */
    .modal {
        display: none;
        position: fixed;
        z-index: 2000;
        left: 0;
        top: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0,0,0,0.8);
    }
    .modal-content {
        position: relative;
        background-color: #2c3e50;
        margin: 2% auto;
        padding: 0;
        width: 90%;
        max-width: 800px;
        border-radius: 10px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.5);
        animation: modalSlideIn 0.3s ease-out;
        border: 1px solid #34495e;
    }
    @keyframes modalSlideIn {
        from { transform: translateY(-50px); opacity: 0; }
        to { transform: translateY(0); opacity: 1; }
    }
    .modal-header {
        padding: 15px 20px;
        background: #34495e;
        border-bottom: 1px solid #4a5f7a;
        border-radius: 10px 10px 0 0;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }
    .modal-header h3 {
        margin: 0;
        color: #ecf0f1;
        font-weight: 600;
    }
    .close-modal {
        background: none;
        border: none;
        font-size: 24px;
        cursor: pointer;
        color: #bdc3c7;
        padding: 0;
        width: 30px;
        height: 30px;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: 50%;
        transition: all 0.3s ease;
    }
    .close-modal:hover {
        background: #4a5f7a;
        color: #ecf0f1;
    }
    .modal-body {
        padding: 0;
    }
    #map-container, #all-orders-map-container {
        width: 100%;
        height: 400px;
        border-radius: 0 0 10px 10px;
    }
    .map-info {
        padding: 15px 20px;
        background: #34495e;
        border-top: 1px solid #4a5f7a;
        color: #ecf0f1;
    }
    .map-info p {
        margin: 5px 0;
        font-size: 14px;
        color: #ecf0f1;
    }
    .map-info strong {
        color: #3498db;
    }
    .map-actions {
        padding: 15px 20px;
        display: flex;
        gap: 10px;
        justify-content: flex-end;
        border-top: 1px solid #4a5f7a;
        background: #2c3e50;
        border-radius: 0 0 10px 10px;
    }
    .btn-small {
        padding: 6px 12px;
        font-size: 13px;
        background: #3498db;
        color: white;
        border: none;
        border-radius: 4px;
        cursor: pointer;
        font-weight: 600;
        transition: background 0.3s ease;
    }
    .btn-small:hover {
        background: #2980b9;
    }
    .btn-small.secondary {
        background: #7f8c8d;
    }
    .btn-small.secondary:hover {
        background: #6c7b7d;
    }
    .fallback-map {
        background: #34495e;
        display: flex;
        align-items: center;
        justify-content: center;
        height: 100%;
        color: #bdc3c7;
        font-size: 16px;
        flex-direction: column;
        gap: 10px;
    }
    .location-status {
        padding: 8px 12px;
        border-radius: 4px;
        font-size: 12px;
        margin: 5px 0;
        font-weight: 600;
    }
    .location-status.success {
        background: #27ae60;
        color: #ecf0f1;
        border: 1px solid #2ecc71;
    }
    .location-status.error {
        background: #c0392b;
        color: #ecf0f1;
        border: 1px solid #e74c3c;
    }
    .location-status.warning {
        background: #f39c12;
        color: #2c3e50;
        border: 1px solid #f1c40f;
    }
    .distance-info {
        background: #3498db;
        padding: 8px 12px;
        border-radius: 4px;
        margin: 5px 0;
        font-size: 13px;
        border-left: 4px solid #2980b9;
        color: white;
        font-weight: 600;
    }
    .orders-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 15px;
    }
    .orders-header h1 {
        margin: 0;
        color: #ecf0f1;
    }

    /* Photo Modal Styles */
    .photo-modal {
        display: none;
        position: fixed;
        z-index: 3000;
        left: 0;
        top: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0,0,0,0.9);
        animation: fadeIn 0.3s ease;
    }
    .photo-modal-content {
        position: relative;
        margin: auto;
        display: block;
        max-width: 90%;
        max-height: 90%;
        margin-top: 2%;
        border-radius: 8px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.5);
        animation: zoomIn 0.3s ease;
    }
    @keyframes fadeIn {
        from { opacity: 0; }
        to { opacity: 1; }
    }
    @keyframes zoomIn {
        from { transform: scale(0.8); opacity: 0; }
        to { transform: scale(1); opacity: 1; }
    }
    .close-photo {
        position: absolute;
        top: 15px;
        right: 35px;
        color: #f1f1f1;
        font-size: 40px;
        font-weight: bold;
        cursor: pointer;
        z-index: 3001;
        background: rgba(0,0,0,0.5);
        width: 50px;
        height: 50px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.3s ease;
    }
    .close-photo:hover {
        background: rgba(0,0,0,0.8);
        transform: scale(1.1);
    }
    .photo-caption {
        text-align: center;
        color: #ccc;
        padding: 10px 20px;
        position: absolute;
        bottom: 0;
        width: 100%;
        background: rgba(0,0,0,0.7);
        border-radius: 0 0 8px 8px;
    }
    .customer-marker {
        cursor: pointer !important;
    }
    .customer-marker img {
        transition: all 0.3s ease;
        border: 3px solid #e74c3c;
    }
    .customer-marker img:hover {
        transform: scale(1.1);
        border-color: #c0392b;
        box-shadow: 0 4px 12px rgba(0,0,0,0.4);
    }

    /* Ensure Leaflet map container has proper dimensions */
    .leaflet-container {
        height: 400px;
        width: 100%;
        border-radius: 0 0 10px 10px;
    }

    /* Additional dark theme improvements */
    .leaflet-popup-content-wrapper {
        background: #2c3e50;
        color: #ecf0f1;
        border-radius: 8px;
        box-shadow: 0 3px 14px rgba(0,0,0,0.4);
    }

    .leaflet-popup-content {
        color: #ecf0f1;
        font-size: 13px;
        line-height: 1.4;
    }

    .leaflet-popup-content h3 {
        color: #3498db;
        margin: 0 0 8px 0;
        font-size: 14px;
    }

    .leaflet-popup-content p {
        color: #bdc3c7;
        margin: 5px 0;
    }

    .leaflet-popup-tip {
        background: #2c3e50;
    }
    

    @media (max-width: 768px) {
        .orders-grid {
            grid-template-columns: repeat(2, 1fr);
            gap: 15px;
        }
        
        .order-card {
            padding: 15px;
        }
        
        .order-card-actions {
            flex-direction: column;
        }
        
        .orders-header {
            flex-direction: column;
            gap: 10px;
            text-align: center;
        }
    }
    
    @media (max-width: 576px) {
        .orders-grid {
            grid-template-columns: 1fr;
            gap: 12px;
        }
        
        .order-card {
            padding: 12px;
        }
        
        .order-card-header {
            flex-direction: column;
            text-align: center;
            gap: 10px;
        }
        
        .order-card-user {
            text-align: center;
        }
    }
    
    /* No orders state */
    .no-orders {
        grid-column: 1 / -1;
        text-align: center;
        padding: 40px 20px;
        color: #bdc3c7;
    }
    
    .no-orders i {
        font-size: 3rem;
        margin-bottom: 15px;
        display: block;
        color: #7f8c8d;
    }
    
    /* Status-specific card accents */
    .order-card.pending {
        border-left: 4px solid #f0ad4e;
    }
    
    .order-card.accepted {
        border-left: 4px solid #5bc0de;
    }
    
    .order-card.delivered {
        border-left: 4px solid #5cb85c;
    }
    
    /* Empty state responsive */
    @media (max-width: 768px) {
        .no-orders {
            padding: 30px 15px;
        }
        
        .no-orders i {
            font-size: 2.5rem;
        }
    }
  </style>
</head>
<body>

  <div class="notification" id="notification"><?php echo t('new_order_available'); ?></div>
  
  <!-- Single Order Map Modal -->
  <div id="mapModal" class="modal">
    <div class="modal-content">
      <div class="modal-header">
        <h3 id="modalTitle"><?php echo t('customer_location'); ?></h3>
        <button class="close-modal" id="closeModal">&times;</button>
      </div>
      <div class="modal-body">
        <div id="map-container">
          <div class="fallback-map" id="fallbackMap">
            <div>🌍 Loading map...</div>
            <div id="locationStatus" class="location-status warning">
              Requesting your location permission...
            </div>
          </div>
        </div>
        <div class="map-info">
          <p><strong id="customerName"></strong></p>
          <p id="customerAddress"></p>
          <p id="customerPhone"></p>
          <p><small>Coordinates: <span id="customerCoords"></span></small></p>
          <div id="distanceInfo" class="distance-info" style="display: none;">
            📏 Distance: <span id="distanceValue">Calculating...</span>
          </div>
          <div id="locationInfo"></div>
        </div>
        <div class="map-actions">
          <button id="refreshLocation" class="btn secondary btn-small">🔄 <?php echo t('refresh_location'); ?></button>
          <button id="openInGoogleMaps" class="btn secondary btn-small"><?php echo t('open_in_google_maps'); ?></button>
          <button id="getDirections" class="btn btn-small">🗺️ <?php echo t('get_directions'); ?></button>
          <button id="closeMap" class="btn secondary btn-small"><?php echo t('close'); ?></button>
        </div>
      </div>
    </div>
  </div>
  
  <!-- All Orders Map Modal -->
  <div id="allOrdersMapModal" class="modal">
    <div class="modal-content">
      <div class="modal-header">
        <h3 id="allOrdersModalTitle"><?php echo t('all_pending_orders_map'); ?> (<span id="pending-count"><?php echo count($pendingOrders); ?></span>)</h3>
        <button class="close-modal" id="closeAllOrdersModal">&times;</button>
      </div>
      <div class="modal-body">
        <div id="all-orders-map-container">
          <div class="fallback-map" id="allOrdersFallbackMap">
            <div>🌍 Loading all orders map...</div>
            <div id="allOrdersLocationStatus" class="location-status warning">
              Requesting your location permission...
            </div>
          </div>
        </div>
        <div class="map-info">
          <p><strong><?php echo t('pending_orders'); ?>: <span id="pending-orders-count"><?php echo count($pendingOrders); ?></span></strong></p>
          <p id="all-orders-info"><?php echo t('click_on_markers_for_order_details'); ?></p>
          <div id="all-orders-distance-info" class="distance-info" style="display: none;">
            📏 Distance to selected order: <span id="all-orders-distance-value">Calculating...</span>
          </div>
        </div>
        <div class="map-actions">
          <button id="refreshAllOrdersLocation" class="btn secondary btn-small">🔄 <?php echo t('refresh_location'); ?></button>
          <button id="closeAllOrdersMap" class="btn secondary btn-small"><?php echo t('close'); ?></button>
        </div>
      </div>
    </div>
  </div>

  <!-- Photo Modal -->
  <div id="photoModal" class="photo-modal">
    <span class="close-photo">&times;</span>
    <img class="photo-modal-content" id="enlargedPhoto">
    <div id="photoCaption" class="photo-caption"></div>
  </div>
  
  <div class="header">
    <div class="header-inner container">
      <div class="brand">
        <div class="logo"></div>
        <h1><?php echo t('driver_panel'); ?> — <?php echo htmlspecialchars($driver['name']); ?></h1>
      </div>
      <div>
        <div class="language-selector">
          <select id="languageSwitch">
            <option value="en" <?php echo $lang == 'en' ? 'selected' : ''; ?>><?php echo t('english'); ?></option>
            <option value="ar" <?php echo $lang == 'ar' ? 'selected' : ''; ?>><?php echo t('arabic'); ?></option>
            <option value="ku" <?php echo $lang == 'ku' ? 'selected' : ''; ?>><?php echo t('kurdish'); ?></option>
          </select>
        </div>
        <span id="connection-status" class="connection-status connected pulse"><?php echo t('live_updates'); ?></span>
        <a class="btn secondary" href="driver_login.php?logout=1"><?php echo t('logout'); ?></a>
      </div>
    </div>
  </div>

  <main class="container">
    <div class="kpis" id="kpis">
      <div class="kpi"><h3><?php echo t('main_stock'); ?></h3><div id="stock-value"><?php echo (int)$driver['stock']; ?></div></div>
      <div class="kpi"><h3><?php echo t('empty'); ?></h3><div id="empty-stock-value"><?php echo (int)$driver['empty_stock']; ?></div></div>
      <div class="kpi"><h3><?php echo t('sold'); ?></h3><div id="sold-value"><?php echo (int)$driver['sold']; ?></div></div>
      <div class="kpi"><h3><?php echo t('orders'); ?></h3><div id="orders-count"><?php echo count($orders); ?></div></div>
      <div class="kpi"><h3><?php echo t('connection_status'); ?></h3><div id="refresh-status"><?php echo t('live_updates'); ?></div></div>
    </div>

    <div class="card" style="margin-top:16px;">
      <div class="orders-header">
        <h1><?php echo t('orders_in_zone'); ?> <span id="orders-badge" class="badge"><?php echo count($orders); ?></span></h1>
        <button class="btn secondary" id="viewAllOrdersMap">
          🗺️ <?php echo t('view_all_orders_map'); ?> (<span id="pending-badge"><?php echo count($pendingOrders); ?></span>)
        </button>
      </div>
      <div class="hr"></div>
      <div class="orders-grid" id="orders-tbody">
        <?php foreach($orders as $o): 
            $pic = !empty($o['profile_pic']) ? "uploads/profile_pics/" . htmlspecialchars($o['profile_pic']) : "assets/images/default-user.png";
            
            $actionHtml = '';
            if ($o['status']=='pending') {
                $actionHtml = '<button class="btn take-order order-card-action" data-order-id="'.$o['id'].'">'.t('take').'</button>';
            } elseif ($o['status']=='accepted' && $o['driver_id']==$driver_id) {
                $actionHtml = '<button class="btn warn mark-delivered order-card-action" data-order-id="'.$o['id'].'">'.t('mark_delivered').'</button>';
            } else {
                $actionHtml = '<span class="muted">—</span>';
            }
            
            $gpsHtml = '';
            if($o['gps_lat'] && $o['gps_lng']) {
                $gpsHtml = '<button class="btn secondary view-map order-card-action" 
                        data-lat="'.$o['gps_lat'].'" 
                        data-lng="'.$o['gps_lng'].'"
                        data-name="'.htmlspecialchars($o['user_name']).'"
                        data-address="'.htmlspecialchars($o['address']).'"
                        data-phone="'.htmlspecialchars($o['user_phone']).'">
                    '.t('view_map').'
                </button>';
            }
        ?>
        <div class="order-card <?php echo $o['status']; ?>" id="order-<?php echo $o['id']; ?>">
            <div class="order-card-header">
                <div class="order-card-user">
                    <div class="order-card-user-name">
                        <img class="profile-pic" src="<?php echo $pic; ?>" alt="Profile">
                        <?php echo htmlspecialchars($o['user_name']); ?>
                    </div>
                    <div class="order-card-user-phone">📞 <?php echo htmlspecialchars($o['user_phone']); ?></div>
                </div>
                <span class="order-card-badge badge <?php echo $o['status']; ?>" id="status-<?php echo $o['id']; ?>"><?php echo t($o['status']); ?></span>
            </div>
            
            <div class="order-card-details">
                <div class="order-card-detail">
                    <i>📍</i>
                    <span class="order-card-address"><?php echo htmlspecialchars($o['address'] ?? t('no_address_provided')); ?></span>
                </div>
                <div class="order-card-detail">
                    <i>🆔</i>
                    <span><?php echo t('order'); ?> #<?php echo $o['id']; ?></span>
                </div>
            </div>
            
            <div class="order-card-actions">
                <?php echo $gpsHtml; ?>
                <?php echo $actionHtml; ?>
            </div>
        </div>
        <?php endforeach; ?>
      </div>
    </div>

    <div class="card" style="margin-top:16px;">
      <h2><?php echo t('manual_stock_management'); ?></h2>
      <div class="hr"></div>
      <form method="post" id="stock-form">
        <div class="grid grid-2">
          <div>
            <label><?php echo t('exchange_empty_cylinders'); ?></label>
            <input type="number" class="input" name="exchange" id="exchange-input" min="0" max="<?php echo $driver['filled_stock']; ?>" placeholder="<?php echo t('sold_cylinders_returned'); ?>">
          </div>
          <div>
            <label><?php echo t('manual_sell'); ?></label>
            <input type="number" class="input" name="manual_sell" id="manual-sell-input" min="0" max="<?php echo $driver['filled_stock']; ?>" placeholder="<?php echo t('sell_filled_cylinders'); ?>">
          </div>
          <div style="grid-column:1 / -1">
            <button type="submit" class="btn"><?php echo t('update_stock'); ?></button>
          </div>
        </div>
      </form>
    </div>

  </main>

  <div class="footer">© <?php echo date('Y'); ?> <?php echo t('propane_service'); ?></div>

  <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>

  <script>
    let lastUpdateTime = Date.now();
    let isPolling = true;
    let currentOrderCount = <?php echo count($orders); ?>;
    let audioContext = null;
    let audioBuffer = null;
    let isAudioReady = false;
    let map = null;
    let allOrdersMap = null;
    let currentMarker = null;
    let customerMarker = null;
    let currentLocation = null;
    let userLocation = null;
    let userLocationMarker = null;
    let allOrdersUserLocationMarker = null;
    let watchId = null;
    let allOrdersWatchId = null;
    let pendingOrdersMarkers = [];
    let selectedOrderMarker = null;
    
    function calculateDistance(lat1, lon1, lat2, lon2) {
        const R = 6371;
        const dLat = (lat2 - lat1) * Math.PI / 180;
        const dLon = (lon2 - lon1) * Math.PI / 180;
        const a = 
            Math.sin(dLat/2) * Math.sin(dLat/2) +
            Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) * 
            Math.sin(dLon/2) * Math.sin(dLon/2);
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
        const distance = R * c;
        return distance;
    }
    
    function formatDistance(distance) {
        if (distance < 1) {
            return (distance * 1000).toFixed(0) + ' meters';
        } else {
            return distance.toFixed(1) + ' km';
        }
    }
    
    function getUserLocation() {
        return new Promise((resolve, reject) => {
            if (!navigator.geolocation) {
                reject(new Error('Geolocation is not supported by this browser.'));
                return;
            }
            
            const options = {
                enableHighAccuracy: true,
                timeout: 10000,
                maximumAge: 60000
            };
            
            navigator.geolocation.getCurrentPosition(
                (position) => {
                    const location = {
                        lat: position.coords.latitude,
                        lng: position.coords.longitude,
                        accuracy: position.coords.accuracy
                    };
                    resolve(location);
                },
                (error) => {
                    let errorMessage = 'Unable to retrieve your location.';
                    switch(error.code) {
                        case error.PERMISSION_DENIED:
                            errorMessage = 'Location access denied. Please allow location access to see your position on the map.';
                            break;
                        case error.POSITION_UNAVAILABLE:
                            errorMessage = 'Location information is unavailable.';
                            break;
                        case error.TIMEOUT:
                            errorMessage = 'Location request timed out.';
                            break;
                    }
                    reject(new Error(errorMessage));
                },
                options
            );
        });
    }
    
    function startWatchingLocation() {
        if (!navigator.geolocation) return;
        
        const options = {
            enableHighAccuracy: true,
            timeout: 15000,
            maximumAge: 30000
        };
        
        watchId = navigator.geolocation.watchPosition(
            (position) => {
                userLocation = {
                    lat: position.coords.latitude,
                    lng: position.coords.longitude,
                    accuracy: position.coords.accuracy
                };
                updateUserLocationOnMap();
            },
            (error) => {
                console.log('Error watching location:', error);
            },
            options
        );
    }
    
    function startWatchingAllOrdersLocation() {
        if (!navigator.geolocation) return;
        
        const options = {
            enableHighAccuracy: true,
            timeout: 15000,
            maximumAge: 30000
        };
        
        allOrdersWatchId = navigator.geolocation.watchPosition(
            (position) => {
                userLocation = {
                    lat: position.coords.latitude,
                    lng: position.coords.longitude,
                    accuracy: position.coords.accuracy
                };
                updateUserLocationOnAllOrdersMap();
            },
            (error) => {
                console.log('Error watching location for all orders map:', error);
            },
            options
        );
    }
    
    function stopWatchingLocation() {
        if (watchId !== null) {
            navigator.geolocation.clearWatch(watchId);
            watchId = null;
        }
    }
    
    function stopWatchingAllOrdersLocation() {
        if (allOrdersWatchId !== null) {
            navigator.geolocation.clearWatch(allOrdersWatchId);
            allOrdersWatchId = null;
        }
    }
    
    function updateUserLocationOnMap() {
        if (!map || !userLocation) return;
        
        if (userLocationMarker) {
            map.removeLayer(userLocationMarker);
        }
        
        const userIcon = L.divIcon({
            className: 'user-location-marker',
            html: '<div style="background:#3498db; width:16px; height:16px; border-radius:50%; border:3px solid white; box-shadow:0 2px 5px rgba(0,0,0,0.3);"></div>',
            iconSize: [22, 22],
            iconAnchor: [11, 11]
        });
        
        if (userLocation.accuracy) {
            L.circle([userLocation.lat, userLocation.lng], {
                color: '#3498db',
                fillColor: '#3498db',
                fillOpacity: 0.1,
                radius: userLocation.accuracy
            }).addTo(map);
        }
        
        userLocationMarker = L.marker([userLocation.lat, userLocation.lng], {icon: userIcon})
            .addTo(map)
            .bindPopup(`
                <div style="padding: 10px; min-width: 200px;">
                    <h3 style="margin: 0 0 8px 0; color: #333;">📍 Your Location</h3>
                    <p style="margin: 0 0 5px 0; color: #666;">Lat: ${userLocation.lat.toFixed(6)}</p>
                    <p style="margin: 0 0 5px 0; color: #666;">Lng: ${userLocation.lng.toFixed(6)}</p>
                    <p style="margin: 0; font-size: 12px; color: #999;">Accuracy: ±${Math.round(userLocation.accuracy)} meters</p>
                </div>
            `);
        
        if (currentLocation) {
            const distance = calculateDistance(
                userLocation.lat, userLocation.lng,
                currentLocation.lat, currentLocation.lng
            );
            
            document.getElementById('distanceValue').textContent = formatDistance(distance);
            document.getElementById('distanceInfo').style.display = 'block';
        }
        
        if (currentLocation) {
            const group = new L.featureGroup([userLocationMarker, currentMarker]);
            map.fitBounds(group.getBounds().pad(0.1));
        }
    }
    
    function updateUserLocationOnAllOrdersMap() {
        if (!allOrdersMap || !userLocation) return;
        
        if (allOrdersUserLocationMarker) {
            allOrdersMap.removeLayer(allOrdersUserLocationMarker);
        }
        
        const userIcon = L.divIcon({
            className: 'user-location-marker',
            html: '<div style="background:#3498db; width:16px; height:16px; border-radius:50%; border:3px solid white; box-shadow:0 2px 5px rgba(0,0,0,0.3);"></div>',
            iconSize: [22, 22],
            iconAnchor: [11, 11]
        });
        
        if (userLocation.accuracy) {
            L.circle([userLocation.lat, userLocation.lng], {
                color: '#3498db',
                fillColor: '#3498db',
                fillOpacity: 0.1,
                radius: userLocation.accuracy
            }).addTo(allOrdersMap);
        }
        
        allOrdersUserLocationMarker = L.marker([userLocation.lat, userLocation.lng], {icon: userIcon})
            .addTo(allOrdersMap)
            .bindPopup(`
                <div style="padding: 10px; min-width: 200px;">
                    <h3 style="margin: 0 0 8px 0; color: #333;">📍 Your Location</h3>
                    <p style="margin: 0 0 5px 0; color: #666;">Lat: ${userLocation.lat.toFixed(6)}</p>
                    <p style="margin: 0 0 5px 0; color: #666;">Lng: ${userLocation.lng.toFixed(6)}</p>
                    <p style="margin: 0; font-size: 12px; color: #999;">Accuracy: ±${Math.round(userLocation.accuracy)} meters</p>
                </div>
            `);
        
        if (selectedOrderMarker && userLocation) {
            const selectedLatLng = selectedOrderMarker.getLatLng();
            const distance = calculateDistance(
                userLocation.lat, userLocation.lng,
                selectedLatLng.lat, selectedLatLng.lng
            );
            
            document.getElementById('all-orders-distance-value').textContent = formatDistance(distance);
            document.getElementById('all-orders-distance-info').style.display = 'block';
        }
        
        if (pendingOrdersMarkers.length > 0) {
            const group = new L.featureGroup([allOrdersUserLocationMarker, ...pendingOrdersMarkers]);
            allOrdersMap.fitBounds(group.getBounds().pad(0.1));
        }
    }
    
    function initMap(lat, lng, name, address, phone, profilePic) {
        try {
            const fallbackMap = document.getElementById('fallbackMap');
            if (fallbackMap) {
                fallbackMap.style.display = 'none';
            }
            
            const mapContainer = document.getElementById('map-container');
            if (!mapContainer) {
                throw new Error('Map container not found');
            }
            
            map = L.map('map-container').setView([lat, lng], 15);
            
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
                maxZoom: 19
            }).addTo(map);
            
            const customerIcon = L.divIcon({
                className: 'customer-marker',
                html: `
                    <div style="position: relative;">
                        <img src="${profilePic}" 
                             style="width: 50px; height: 50px; border-radius: 50%; border: 3px solid #e74c3c; cursor: pointer; box-shadow: 0 2px 8px rgba(0,0,0,0.3); object-fit: cover;"
                             onclick="showProfileModal('${profilePic}', '${escapeHtml(name)}', '${escapeHtml(phone)}')"
                             alt="${escapeHtml(name)}">
                        <div style="position: absolute; bottom: -5px; right: -5px; background: #e74c3c; color: white; border-radius: 50%; width: 20px; height: 20px; display: flex; align-items: center; justify-content: center; font-size: 12px; border: 2px solid white; box-shadow: 0 1px 3px rgba(0,0,0,0.3);">📍</div>
                    </div>
                `,
                iconSize: [60, 60],
                iconAnchor: [30, 55]
            });
            
            currentMarker = L.marker([lat, lng], {icon: customerIcon}).addTo(map)
                .bindPopup(`
                    <div style="padding: 12px; min-width: 220px; max-width: 280px;">
                        <div style="text-align: center; margin-bottom: 12px;">
                            <img src="${profilePic}" 
                                 style="width: 70px; height: 70px; border-radius: 50%; border: 3px solid #e74c3c; cursor: pointer; margin-bottom: 8px; object-fit: cover;" 
                                 onclick="showProfileModal('${profilePic}', '${escapeHtml(name)}', '${escapeHtml(phone)}')"
                                 alt="${escapeHtml(name)}">
                            <h3 style="margin: 0 0 5px 0; color: #2c3e50; font-size: 16px; font-weight: 600;">${escapeHtml(name)}</h3>
                            <p style="margin: 0; color: #7f8c8d; font-size: 13px;">📞 ${escapeHtml(phone)}</p>
                        </div>
                        <div style="background: #f8f9fa; padding: 8px; border-radius: 4px; margin-bottom: 10px;">
                            <p style="margin: 0; color: #34495e; font-size: 13px; line-height: 1.4;">📍 ${escapeHtml(address || 'No address provided')}</p>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <small style="color: #95a5a6;">Coordinates: ${lat.toFixed(6)}, ${lng.toFixed(6)}</small>
                            <button onclick="showProfileModal('${profilePic}', '${escapeHtml(name)}', '${escapeHtml(phone)}')" 
                                    style="background: #3498db; color: white; border: none; padding: 4px 8px; border-radius: 3px; font-size: 11px; cursor: pointer;">
                                👁️ View Photo
                            </button>
                        </div>
                    </div>
                `)
                .openPopup();
            
            getUserLocation().then(location => {
                userLocation = location;
                updateUserLocationOnMap();
                startWatchingLocation();
                
                document.getElementById('locationStatus').className = 'location-status success';
                document.getElementById('locationStatus').innerHTML = '✅ Location access granted - tracking your position';
            }).catch(error => {
                console.error('Error getting location:', error);
                document.getElementById('locationStatus').className = 'location-status error';
                document.getElementById('locationStatus').innerHTML = '❌ ' + error.message;
            });
            
        } catch (error) {
            console.error('Map initialization error:', error);
            const fallbackMap = document.getElementById('fallbackMap');
            if (fallbackMap) {
                fallbackMap.innerHTML = `
                    <div style="text-align: center;">
                        <p>🌍 Map Loading Failed</p>
                        <p><small>Coordinates: ${lat}, ${lng}</small></p>
                        <p><small>${error.message}</small></p>
                        <button onclick="openInGoogleMaps()" class="btn btn-small" style="margin-top: 10px;">Open in Google Maps</button>
                    </div>
                `;
            }
            throw error;
        }
    }
    
    function initAllOrdersMap(orders) {
        try {
            const fallbackMap = document.getElementById('allOrdersFallbackMap');
            if (fallbackMap) {
                fallbackMap.style.display = 'none';
            }
            
            const mapContainer = document.getElementById('all-orders-map-container');
            if (!mapContainer) {
                throw new Error('All orders map container not found');
            }
            
            if (allOrdersMap) {
                allOrdersMap.remove();
            }
            
            pendingOrdersMarkers = [];
            selectedOrderMarker = null;
            
            const pendingOrders = orders.filter(order => order.status === 'pending');
            
            document.getElementById('pending-count').textContent = pendingOrders.length;
            document.getElementById('pending-orders-count').textContent = pendingOrders.length;
            
            if (pendingOrders.length === 0) {
                document.getElementById('allOrdersFallbackMap').style.display = 'flex';
                document.getElementById('allOrdersFallbackMap').innerHTML = `
                    <div style="text-align: center;">
                        <p>📭 No pending orders available</p>
                        <p><small>There are currently no pending orders in your area.</small></p>
                    </div>
                `;
                return;
            }
            
            let centerLat = 0, centerLng = 0;
            let validCoords = 0;
            
            pendingOrders.forEach(order => {
                if (order.gps_lat && order.gps_lng) {
                    centerLat += parseFloat(order.gps_lat);
                    centerLng += parseFloat(order.gps_lng);
                    validCoords++;
                }
            });
            
            if (validCoords > 0) {
                centerLat /= validCoords;
                centerLng /= validCoords;
            } else {
                centerLat = 36.0;
                centerLng = 44.0;
            }
            
            allOrdersMap = L.map('all-orders-map-container').setView([centerLat, centerLng], 12);
            
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
                maxZoom: 19
            }).addTo(allOrdersMap);
            
            pendingOrders.forEach(order => {
                if (order.gps_lat && order.gps_lng) {
                    const lat = parseFloat(order.gps_lat);
                    const lng = parseFloat(order.gps_lng);
                    const profilePic = order.profile_pic ? 
                        `uploads/profile_pics/${order.profile_pic}` : 
                        'assets/images/default-user.png';
                    
                    const pendingIcon = L.divIcon({
                        className: 'pending-order-marker',
                        html: `
                            <div style="position: relative;">
                                <img src="${profilePic}" 
                                     style="width: 40px; height: 40px; border-radius: 50%; border: 3px solid #f39c12; cursor: pointer; box-shadow: 0 2px 6px rgba(0,0,0,0.3); object-fit: cover;"
                                     onclick="showProfileModal('${profilePic}', '${escapeHtml(order.user_name)}', '${escapeHtml(order.user_phone)}')"
                                     alt="${escapeHtml(order.user_name)}">
                                <div style="position: absolute; bottom: -3px; right: -3px; background: #f39c12; color: white; border-radius: 50%; width: 16px; height: 16px; display: flex; align-items: center; justify-content: center; font-size: 10px; border: 2px solid white; box-shadow: 0 1px 3px rgba(0,0,0,0.3);">📍</div>
                            </div>
                        `,
                        iconSize: [46, 46],
                        iconAnchor: [23, 46]
                    });
                    
                    const marker = L.marker([lat, lng], {icon: pendingIcon})
                        .addTo(allOrdersMap)
                        .bindPopup(`
                            <div style="padding: 12px; min-width: 220px; max-width: 280px;">
                                <div style="text-align: center; margin-bottom: 10px;">
                                    <img src="${profilePic}" 
                                         style="width: 60px; height: 60px; border-radius: 50%; border: 3px solid #f39c12; cursor: pointer; margin-bottom: 8px; object-fit: cover;" 
                                         onclick="showProfileModal('${profilePic}', '${escapeHtml(order.user_name)}', '${escapeHtml(order.user_phone)}')"
                                         alt="${escapeHtml(order.user_name)}">
                                    <h3 style="margin: 0 0 5px 0; color: #2c3e50; font-size: 15px; font-weight: 600;">${escapeHtml(order.user_name)}</h3>
                                    <p style="margin: 0; color: #7f8c8d; font-size: 12px;">📞 ${escapeHtml(order.user_phone)}</p>
                                </div>
                                <div style="background: #f8f9fa; padding: 6px; border-radius: 4px; margin-bottom: 10px;">
                                    <p style="margin: 0; color: #34495e; font-size: 12px; line-height: 1.4;">📍 ${escapeHtml(order.address || 'No address provided')}</p>
                                </div>
                                <p style="margin: 0 0 8px 0; font-size: 11px; color: #95a5a6;">Order #${order.id}</p>
                                <button class="btn take-order-from-map" data-order-id="${order.id}" style="width:100%; padding:6px; font-size:12px; background:#27ae60; color:white; border:none; border-radius:4px; cursor:pointer;">
                                    ✅ Take This Order
                                </button>
                            </div>
                        `);
                    
                    marker.on('click', function() {
                        if (selectedOrderMarker) {
                            const oldProfilePic = selectedOrderMarker.options.icon.options.html.match(/src="([^"]*)"/)[1];
                            const oldName = selectedOrderMarker.options.icon.options.html.match(/alt="([^"]*)"/)[1];
                            const selectedIcon = L.divIcon({
                                className: 'pending-order-marker',
                                html: `
                                    <div style="position: relative;">
                                        <img src="${oldProfilePic}" 
                                             style="width: 40px; height: 40px; border-radius: 50%; border: 3px solid #f39c12; cursor: pointer; box-shadow: 0 2px 6px rgba(0,0,0,0.3); object-fit: cover;"
                                             onclick="showProfileModal('${oldProfilePic}', '${oldName}', '')"
                                             alt="${oldName}">
                                        <div style="position: absolute; bottom: -3px; right: -3px; background: #f39c12; color: white; border-radius: 50%; width: 16px; height: 16px; display: flex; align-items: center; justify-content: center; font-size: 10px; border: 2px solid white; box-shadow: 0 1px 3px rgba(0,0,0,0.3);">📍</div>
                                    </div>
                                `,
                                iconSize: [46, 46],
                                iconAnchor: [23, 46]
                            });
                            selectedOrderMarker.setIcon(selectedIcon);
                        }
                        
                        const selectedIcon = L.divIcon({
                            className: 'selected-order-marker',
                            html: `
                                <div style="position: relative;">
                                    <img src="${profilePic}" 
                                         style="width: 50px; height: 50px; border-radius: 50%; border: 3px solid #e74c3c; cursor: pointer; box-shadow: 0 3px 10px rgba(0,0,0,0.4); object-fit: cover;"
                                         onclick="showProfileModal('${profilePic}', '${escapeHtml(order.user_name)}', '${escapeHtml(order.user_phone)}')"
                                         alt="${escapeHtml(order.user_name)}">
                                    <div style="position: absolute; bottom: -4px; right: -4px; background: #e74c3c; color: white; border-radius: 50%; width: 18px; height: 18px; display: flex; align-items: center; justify-content: center; font-size: 10px; border: 2px solid white; box-shadow: 0 1px 3px rgba(0,0,0,0.3);">📍</div>
                                </div>
                            `,
                            iconSize: [56, 56],
                            iconAnchor: [28, 56]
                        });
                        
                        marker.setIcon(selectedIcon);
                        selectedOrderMarker = marker;
                        
                        if (userLocation) {
                            const distance = calculateDistance(
                                userLocation.lat, userLocation.lng,
                                lat, lng
                            );
                            
                            document.getElementById('all-orders-distance-value').textContent = formatDistance(distance);
                            document.getElementById('all-orders-distance-info').style.display = 'block';
                        }
                    });
                    
                    pendingOrdersMarkers.push(marker);
                }
            });
            
            getUserLocation().then(location => {
                userLocation = location;
                updateUserLocationOnAllOrdersMap();
                startWatchingAllOrdersLocation();
                
                document.getElementById('allOrdersLocationStatus').className = 'location-status success';
                document.getElementById('allOrdersLocationStatus').innerHTML = '✅ Location access granted - tracking your position';
            }).catch(error => {
                console.error('Error getting location for all orders map:', error);
                document.getElementById('allOrdersLocationStatus').className = 'location-status error';
                document.getElementById('allOrdersLocationStatus').innerHTML = '❌ ' + error.message;
                
                if (pendingOrdersMarkers.length > 0) {
                    const group = new L.featureGroup(pendingOrdersMarkers);
                    allOrdersMap.fitBounds(group.getBounds().pad(0.1));
                }
            });
            
        } catch (error) {
            console.error('All orders map initialization error:', error);
            const fallbackMap = document.getElementById('allOrdersFallbackMap');
            if (fallbackMap) {
                fallbackMap.innerHTML = `
                    <div style="text-align: center;">
                        <p>🌍 Map Loading Failed</p>
                        <p><small>${error.message}</small></p>
                    </div>
                `;
            }
        }
    }
    
    function showProfileModal(photoSrc, name, phone) {
        const modal = document.getElementById('photoModal');
        const enlargedImg = document.getElementById('enlargedPhoto');
        const caption = document.getElementById('photoCaption');
        
        enlargedImg.src = photoSrc;
        enlargedImg.alt = name;
        caption.innerHTML = `<strong>${escapeHtml(name)}</strong><br>📞 ${escapeHtml(phone)}`;
        
        modal.style.display = 'block';
        
        document.querySelector('.close-photo').onclick = function() {
            modal.style.display = 'none';
        };
        
        modal.onclick = function(event) {
            if (event.target === modal) {
                modal.style.display = 'none';
            }
        };
        
        document.addEventListener('keydown', function closeOnEscape(event) {
            if (event.key === 'Escape') {
                modal.style.display = 'none';
                document.removeEventListener('keydown', closeOnEscape);
            }
        });
    }
    
    function showMapModal(lat, lng, name, address, phone, profilePic) {
        currentLocation = {lat: parseFloat(lat), lng: parseFloat(lng)};
        
        document.getElementById('modalTitle').textContent = name + ' - <?php echo t('location'); ?>';
        document.getElementById('customerName').textContent = name;
        document.getElementById('customerAddress').textContent = address || 'No address provided';
        document.getElementById('customerPhone').textContent = phone;
        document.getElementById('customerCoords').textContent = lat + ', ' + lng;
        document.getElementById('distanceInfo').style.display = 'none';
        
        const fallbackMap = document.getElementById('fallbackMap');
        if (fallbackMap) {
            fallbackMap.style.display = 'flex';
            fallbackMap.innerHTML = `
                <div>🌍 Loading map...</div>
                <div id="locationStatus" class="location-status warning">
                    Requesting your location permission...
                </div>
            `;
        }
        
        if (map) {
            stopWatchingLocation();
            map.remove();
            map = null;
        }
        currentMarker = null;
        userLocationMarker = null;
        userLocation = null;
        
        setTimeout(() => {
            try {
                initMap(parseFloat(lat), parseFloat(lng), name, address, phone, profilePic);
            } catch (error) {
                console.error('Failed to initialize map:', error);
            }
        }, 100);
        
        document.getElementById('mapModal').style.display = 'block';
    }
    
    function showAllOrdersMapModal() {
        const fallbackMap = document.getElementById('allOrdersFallbackMap');
        if (fallbackMap) {
            fallbackMap.style.display = 'flex';
            fallbackMap.innerHTML = `
                <div>🌍 Loading all orders map...</div>
                <div id="allOrdersLocationStatus" class="location-status warning">
                    Requesting your location permission...
                </div>
            `;
        }
        
        if (allOrdersMap) {
            stopWatchingAllOrdersLocation();
            allOrdersMap.remove();
            allOrdersMap = null;
        }
        pendingOrdersMarkers = [];
        selectedOrderMarker = null;
        allOrdersUserLocationMarker = null;
        userLocation = null;
        
        fetch(`?ajax=get_updates&t=${Date.now()}`)
            .then(response => response.json())
            .then(data => {
                setTimeout(() => {
                    try {
                        initAllOrdersMap(data.orders);
                    } catch (error) {
                        console.error('Failed to initialize all orders map:', error);
                    }
                }, 100);
            })
            .catch(error => {
                console.error('Error fetching orders for map:', error);
            });
        
        document.getElementById('allOrdersMapModal').style.display = 'block';
    }
    
    function closeMapModal() {
        document.getElementById('mapModal').style.display = 'none';
        stopWatchingLocation();
        if (map) {
            try {
                map.remove();
            } catch (e) {
                console.log('Error removing map:', e);
            }
            map = null;
        }
        currentMarker = null;
        userLocationMarker = null;
        userLocation = null;
    }
    
    function closeAllOrdersMapModal() {
        document.getElementById('allOrdersMapModal').style.display = 'none';
        stopWatchingAllOrdersLocation();
        if (allOrdersMap) {
            try {
                allOrdersMap.remove();
            } catch (e) {
                console.log('Error removing all orders map:', e);
            }
            allOrdersMap = null;
        }
        pendingOrdersMarkers = [];
        selectedOrderMarker = null;
        allOrdersUserLocationMarker = null;
        userLocation = null;
    }
    
    function refreshLocation() {
        if (!map) return;
        
        document.getElementById('locationStatus').className = 'location-status warning';
        document.getElementById('locationStatus').innerHTML = '🔄 Refreshing your location...';
        
        getUserLocation().then(location => {
            userLocation = location;
            updateUserLocationOnMap();
            document.getElementById('locationStatus').className = 'location-status success';
            document.getElementById('locationStatus').innerHTML = '✅ Location updated successfully';
        }).catch(error => {
            document.getElementById('locationStatus').className = 'location-status error';
            document.getElementById('locationStatus').innerHTML = '❌ ' + error.message;
        });
    }
    
    function refreshAllOrdersLocation() {
        if (!allOrdersMap) return;
        
        document.getElementById('allOrdersLocationStatus').className = 'location-status warning';
        document.getElementById('allOrdersLocationStatus').innerHTML = '🔄 Refreshing your location...';
        
        getUserLocation().then(location => {
            userLocation = location;
            updateUserLocationOnAllOrdersMap();
            document.getElementById('allOrdersLocationStatus').className = 'location-status success';
            document.getElementById('allOrdersLocationStatus').innerHTML = '✅ Location updated successfully';
        }).catch(error => {
            document.getElementById('allOrdersLocationStatus').className = 'location-status error';
            document.getElementById('allOrdersLocationStatus').innerHTML = '❌ ' + error.message;
        });
    }
    
    function openInGoogleMaps() {
        if (currentLocation) {
            const url = `https://www.google.com/maps?q=${currentLocation.lat},${currentLocation.lng}`;
            window.open(url, '_blank');
        }
    }
    
    function getDirections() {
        if (userLocation && currentLocation) {
            const url = `https://www.google.com/maps/dir/${userLocation.lat},${userLocation.lng}/${currentLocation.lat},${currentLocation.lng}`;
            window.open(url, '_blank');
        } else if (currentLocation) {
            openInGoogleMaps();
        }
    }
    
    function initAudio() {
        try {
            audioContext = new (window.AudioContext || window.webkitAudioContext)();
            loadSound('order.mp3');
        } catch (error) {
            console.error('Audio initialization failed:', error);
        }
    }
    
    function loadSound(url) {
        fetch(url)
            .then(response => response.arrayBuffer())
            .then(arrayBuffer => audioContext.decodeAudioData(arrayBuffer))
            .then(buffer => {
                audioBuffer = buffer;
                isAudioReady = true;
            })
            .catch(error => {
                console.error('Sound loading failed:', error);
            });
    }
    
    function playOrderSound() {
        if (!isAudioReady || !audioContext || !audioBuffer) {
            console.log('Audio not ready yet');
            return;
        }
        
        try {
            if (audioContext.state === 'suspended') {
                audioContext.resume();
            }
            
            const source = audioContext.createBufferSource();
            source.buffer = audioBuffer;
            source.connect(audioContext.destination);
            source.start(0);
            
            console.log('Sound played successfully');
        } catch (error) {
            console.error('Error playing sound:', error);
        }
    }
    
    function playSimpleSound() {
        try {
            const audio = new Audio('assets/sounds/order.mp3');
            audio.play().catch(e => {
                console.log('Simple audio play failed:', e);
                playOrderSound();
            });
        } catch (error) {
            console.error('Simple audio failed:', error);
        }
    }
    
    function showNotification(message) {
      const notification = document.getElementById('notification');
      notification.textContent = message;
      notification.style.display = 'block';
      
      playSimpleSound();
      
      setTimeout(() => {
        notification.style.display = 'none';
      }, 5000);
    }
    
    function updateUI(data) {
      document.getElementById('stock-value').textContent = data.driver.stock || 0;
      document.getElementById('empty-stock-value').textContent = data.driver.empty_stock || 0;
      document.getElementById('sold-value').textContent = data.driver.sold || 0;
      
      const pendingOrdersCount = data.orders.filter(order => order.status === 'pending').length;
      document.getElementById('pending-badge').textContent = pendingOrdersCount;
      
      if (data.orders.length > currentOrderCount) {
        showNotification('<?php echo t('new_order_available_in_zone'); ?>');
      }
      
      currentOrderCount = data.orders.length;
      
      document.getElementById('orders-count').textContent = data.orders.length;
      document.getElementById('orders-badge').textContent = data.orders.length;
      
      updateOrdersTable(data.orders);
    }
    
    function updateOrdersTable(orders) {
      const container = document.getElementById('orders-tbody');
      
      container.innerHTML = '';
      
      if (orders.length === 0) {
        container.innerHTML = `
            <div class="no-orders">
                <i>📭</i>
                <h3><?php echo t('no_orders_available'); ?></h3>
                <p><?php echo t('no_orders_in_your_area'); ?></p>
            </div>
        `;
        return;
      }
      
      orders.forEach(order => {
        const pic = order.profile_pic ? 
          `uploads/profile_pics/${order.profile_pic}` : 
          'assets/images/default-user.png';
        
        let actionHtml = '';
        if (order.status === 'pending') {
          actionHtml = `<button class="btn take-order order-card-action" data-order-id="${order.id}"><?php echo t('take'); ?></button>`;
        } else if (order.status === 'accepted' && order.driver_id == <?php echo $driver_id; ?>) {
          actionHtml = `<button class="btn warn mark-delivered order-card-action" data-order-id="${order.id}"><?php echo t('mark_delivered'); ?></button>`;
        } else {
          actionHtml = '<span class="muted">—</span>';
        }
        
        const gpsHtml = order.gps_lat && order.gps_lng ? 
          `<button class="btn secondary view-map order-card-action" 
                  data-lat="${order.gps_lat}" 
                  data-lng="${order.gps_lng}"
                  data-name="${escapeHtml(order.user_name)}"
                  data-address="${escapeHtml(order.address || '')}"
                  data-phone="${escapeHtml(order.user_phone)}">
            <?php echo t('view_map'); ?>
          </button>` :
          '';
        
        const card = document.createElement('div');
        card.className = `order-card ${order.status}`;
        card.id = `order-${order.id}`;
        card.innerHTML = `
            <div class="order-card-header">
                <div class="order-card-user">
                    <div class="order-card-user-name">
                        <img class="profile-pic" src="${pic}" alt="Profile">
                        ${escapeHtml(order.user_name)}
                    </div>
                    <div class="order-card-user-phone">📞 ${escapeHtml(order.user_phone)}</div>
                </div>
                <span class="order-card-badge badge ${order.status}" id="status-${order.id}">${t(order.status)}</span>
            </div>
            
            <div class="order-card-details">
                <div class="order-card-detail">
                    <i>📍</i>
                    <span class="order-card-address">${escapeHtml(order.address || '<?php echo t('no_address_provided'); ?>')}</span>
                </div>
                <div class="order-card-detail">
                    <i>🆔</i>
                    <span><?php echo t('order'); ?> #${order.id}</span>
                </div>
            </div>
            
            <div class="order-card-actions">
                ${gpsHtml}
                ${actionHtml}
            </div>
        `;
        
        container.appendChild(card);
      });
      
      attachEventListeners();
    }
    
    function attachEventListeners() {
      document.querySelectorAll('.take-order').forEach(button => {
        button.addEventListener('click', function(e) {
          e.preventDefault();
          const orderId = this.getAttribute('data-order-id');
          takeOrder(orderId);
        });
      });
      
      document.querySelectorAll('.mark-delivered').forEach(button => {
        button.addEventListener('click', function(e) {
          e.preventDefault();
          const orderId = this.getAttribute('data-order-id');
          markOrderDelivered(orderId);
        });
      });
      
      document.querySelectorAll('.view-map').forEach(button => {
        button.addEventListener('click', function(e) {
          e.preventDefault();
          const lat = this.getAttribute('data-lat');
          const lng = this.getAttribute('data-lng');
          const name = this.getAttribute('data-name');
          const address = this.getAttribute('data-address');
          const phone = this.getAttribute('data-phone');
          const profilePic = this.closest('.order-card').querySelector('.profile-pic').src;
          showMapModal(lat, lng, name, address, phone, profilePic);
        });
      });
      
      document.addEventListener('click', function(e) {
        if (e.target && e.target.classList.contains('take-order-from-map')) {
          e.preventDefault();
          const orderId = e.target.getAttribute('data-order-id');
          takeOrder(orderId);
          closeAllOrdersMapModal();
        }
      });
    }
    
    function takeOrder(orderId) {
      fetch(`?ajax=take_order&order_id=${orderId}`)
        .then(response => response.json())
        .then(result => {
          if (result.success) {
            checkForUpdates();
          } else {
            alert('Failed to take order: ' + result.message);
          }
        })
        .catch(error => {
          console.error('Error taking order:', error);
          alert('Error taking order');
        });
    }
    
    function markOrderDelivered(orderId) {
      fetch(`?ajax=mark_delivered&order_id=${orderId}`)
        .then(response => response.json())
        .then(result => {
          if (result.success) {
            checkForUpdates();
          } else {
            alert('Failed to mark order as delivered: ' + result.message);
          }
        })
        .catch(error => {
          console.error('Error marking order as delivered:', error);
          alert('Error marking order as delivered');
        });
    }
    
    function checkForUpdates() {
      if (!isPolling) return;
      
      fetch(`?ajax=get_updates&t=${Date.now()}`)
        .then(response => {
          if (!response.ok) {
            throw new Error('Network response was not ok');
          }
          return response.json();
        })
        .then(data => {
          updateUI(data);
          lastUpdateTime = Date.now();
          updateConnectionStatus(true);
        })
        .catch(error => {
          console.error('Error fetching updates:', error);
          updateConnectionStatus(false);
        })
        .finally(() => {
          setTimeout(checkForUpdates, 3000);
        });
    }
    
    function updateConnectionStatus(connected) {
      const statusElement = document.getElementById('connection-status');
      if (connected) {
        statusElement.textContent = '<?php echo t('live_updates'); ?>';
        statusElement.className = 'connection-status connected pulse';
      } else {
        statusElement.textContent = '<?php echo t('disconnected'); ?>';
        statusElement.className = 'connection-status disconnected';
        statusElement.classList.remove('pulse');
      }
    }
    
    function escapeHtml(unsafe) {
      return unsafe
        .replace(/&/g, "&amp;")
        .replace(/</g, "&lt;")
        .replace(/>/g, "&gt;")
        .replace(/"/g, "&quot;")
        .replace(/'/g, "&#039;");
    }
    
    const translations = {
        'en': <?php echo json_encode($texts['en']); ?>,
        'ar': <?php echo json_encode($texts['ar']); ?>,
        'ku': <?php echo json_encode($texts['ku']); ?>
    };
    
    function t(key) {
        const lang = document.documentElement.lang;
        return translations[lang] && translations[lang][key] ? translations[lang][key] : key;
    }
    
    document.addEventListener('DOMContentLoaded', function() {
      initAudio();
      
      document.getElementById('closeModal').addEventListener('click', closeMapModal);
      document.getElementById('closeMap').addEventListener('click', closeMapModal);
      document.getElementById('refreshLocation').addEventListener('click', refreshLocation);
      document.getElementById('openInGoogleMaps').addEventListener('click', openInGoogleMaps);
      document.getElementById('getDirections').addEventListener('click', getDirections);
      
      document.getElementById('closeAllOrdersModal').addEventListener('click', closeAllOrdersMapModal);
      document.getElementById('closeAllOrdersMap').addEventListener('click', closeAllOrdersMapModal);
      document.getElementById('refreshAllOrdersLocation').addEventListener('click', refreshAllOrdersLocation);
      
      document.getElementById('viewAllOrdersMap').addEventListener('click', showAllOrdersMapModal);
      
      document.getElementById('mapModal').addEventListener('click', function(e) {
        if (e.target === this) {
          closeMapModal();
        }
      });
      
      document.getElementById('allOrdersMapModal').addEventListener('click', function(e) {
        if (e.target === this) {
          closeAllOrdersMapModal();
        }
      });
      
      document.querySelector('.close-photo').addEventListener('click', function() {
        document.getElementById('photoModal').style.display = 'none';
      });
      
      checkForUpdates();
      
      attachEventListeners();
      
      document.getElementById('stock-form').addEventListener('submit', function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        
        fetch('?ajax=update_stock', {
          method: 'POST',
          body: formData
        })
        .then(response => response.json())
        .then(result => {
          if (result.success) {
            checkForUpdates();
            document.getElementById('exchange-input').value = '';
            document.getElementById('manual-sell-input').value = '';
          } else {
            alert('Failed to update stock: ' + result.message);
          }
        })
        .catch(error => {
          console.error('Error updating stock:', error);
          alert('Error updating stock');
        });
      });
      
      document.getElementById('languageSwitch').addEventListener('change', function() {
        window.location.href = '?lang=' + this.value;
      });
      
      document.addEventListener('visibilitychange', function() {
        if (document.hidden) {
          isPolling = false;
          stopWatchingLocation();
          stopWatchingAllOrdersLocation();
        } else {
          isPolling = true;
          checkForUpdates();
          if (map && userLocation) {
            startWatchingLocation();
          }
          if (allOrdersMap && userLocation) {
            startWatchingAllOrdersLocation();
          }
        }
      });
      
      document.addEventListener('click', function() {
        if (audioContext && audioContext.state === 'suspended') {
          audioContext.resume();
        }
      }, { once: true });
    });
  </script>
</body>
</html>