<?php
session_start();
require 'db.php';
require 'lang.php'; // Include the language file

if (!isset($_SESSION['driver_id'])) { 
    header("Location: driver_login.php"); 
    exit; 
}
$driver_id = $_SESSION['driver_id'];

// Handle AJAX requests
if (isset($_GET['ajax'])) {
    header('Content-Type: application/json');
    
    if ($_GET['ajax'] == 'take_order' && isset($_GET['order_id'])) {
        $order_id = (int)$_GET['order_id'];
        $stmt = $pdo->prepare("UPDATE orders SET driver_id=?, status='accepted' WHERE id=? AND status='pending'");
        $success = $stmt->execute([$driver_id, $order_id]);
        
        if ($success) {
            echo json_encode(['success' => true]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Order no longer available']);
        }
        exit;
    }
    
    if ($_GET['ajax'] == 'mark_delivered' && isset($_GET['order_id'])) {
        $order_id = (int)$_GET['order_id'];
        
        $pdo->beginTransaction();
        try {
            $stmt = $pdo->prepare("UPDATE orders SET status='delivered' WHERE id=? AND driver_id=?");
            $stmt->execute([$order_id, $driver_id]);
            
            $stmt2 = $pdo->prepare("UPDATE drivers SET empty_stock = IFNULL(empty_stock,0) + 1 WHERE id=?");
            $stmt2->execute([$driver_id]);
            
            $pdo->commit();
            echo json_encode(['success' => true]);
        } catch (Exception $e) {
            $pdo->rollBack();
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
        exit;
    }
    
    if ($_GET['ajax'] == 'update_stock') {
        $manual_sell = (int)($_POST['manual_sell'] ?? 0);
        $exchange = (int)($_POST['exchange'] ?? 0);

        if ($manual_sell > 0) {
            $stmt = $pdo->prepare("UPDATE drivers 
                                   SET stock = GREATEST(IFNULL(stock,0) - ?,0),
                                       sold = IFNULL(sold,0) + ?
                                   WHERE id=?");
            $stmt->execute([$manual_sell, $manual_sell, $driver_id]);
        }

        if ($exchange > 0) {
            $stmt = $pdo->prepare("UPDATE drivers SET empty_stock = IFNULL(empty_stock,0) + ? WHERE id=?");
            $stmt->execute([$exchange, $driver_id]);
        }

        $stmt = $pdo->prepare("UPDATE drivers 
                               SET filled_stock = GREATEST(IFNULL(stock,0) - IFNULL(sold,0) - IFNULL(empty_stock,0),0)
                               WHERE id=?");
        $stmt->execute([$driver_id]);
        
        echo json_encode(['success' => true]);
        exit;
    }
    
    if ($_GET['ajax'] == 'get_updates') {
        // Fetch updated driver info
        $stmt = $pdo->prepare("SELECT * FROM drivers WHERE id=?");
        $stmt->execute([$driver_id]);
        $driver = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Fetch updated orders
        $ordersStmt = $pdo->prepare("SELECT o.*, u.name AS user_name, u.phone AS user_phone, u.address, u.gps_lat, u.gps_lng, u.profile_pic
                                     FROM orders o 
                                     JOIN users u ON o.user_id=u.id 
                                     WHERE o.area_id=? 
                                     ORDER BY o.created_at DESC");
        $ordersStmt->execute([$driver['area_id']]);
        $orders = $ordersStmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'driver' => $driver,
            'orders' => $orders
        ]);
        exit;
    }
}

// Handle regular form submissions (non-AJAX)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !isset($_GET['ajax'])) {
    $manual_sell = (int)($_POST['manual_sell'] ?? 0);
    $exchange = (int)($_POST['exchange'] ?? 0);

    if ($manual_sell > 0) {
        $stmt = $pdo->prepare("UPDATE drivers 
                               SET stock = GREATEST(IFNULL(stock,0) - ?,0),
                                   sold = IFNULL(sold,0) + ?
                               WHERE id=?");
        $stmt->execute([$manual_sell, $manual_sell, $driver_id]);
    }

    if ($exchange > 0) {
        $stmt = $pdo->prepare("UPDATE drivers SET empty_stock = IFNULL(empty_stock,0) + ? WHERE id=?");
        $stmt->execute([$exchange, $driver_id]);
    }

    $stmt = $pdo->prepare("UPDATE drivers 
                           SET filled_stock = GREATEST(IFNULL(stock,0) - IFNULL(sold,0) - IFNULL(empty_stock,0),0)
                           WHERE id=?");
    $stmt->execute([$driver_id]);

    header("Location: driver_panel.php");
    exit;
}

// Handle regular GET requests (non-AJAX)
if (isset($_GET['take']) && !isset($_GET['ajax'])) {
    $stmt = $pdo->prepare("UPDATE orders SET driver_id=?, status='accepted' WHERE id=? AND status='pending'");
    $stmt->execute([$driver_id, (int)$_GET['take']]);
    header("Location: driver_panel.php");
    exit;
}

if (isset($_GET['delivered']) && !isset($_GET['ajax'])) {
    $orderId = (int)$_GET['delivered'];
    $stmtCheck = $pdo->prepare("SELECT status FROM orders WHERE id=? AND driver_id=?");
    $stmtCheck->execute([$orderId, $driver_id]);
    $orderStatus = $stmtCheck->fetchColumn();
    if ($orderStatus === 'accepted') {
        $pdo->beginTransaction();
        try {
            $stmt = $pdo->prepare("UPDATE orders SET status='delivered' WHERE id=? AND driver_id=?");
            $stmt->execute([$orderId, $driver_id]);
            $stmt2 = $pdo->prepare("UPDATE drivers SET empty_stock = IFNULL(empty_stock,0) + 1 WHERE id=?");
            $stmt2->execute([$driver_id]);
            $pdo->commit();
            header("Location: driver_panel.php");
            exit;
        } catch (Exception $e) {
            $pdo->rollBack();
            echo "Error: " . $e->getMessage();
        }
    } else {
        header("Location: driver_panel.php");
        exit;
    }
}

// Fetch driver info
$stmt = $pdo->prepare("SELECT * FROM drivers WHERE id=?");
$stmt->execute([$driver_id]);
$driver = $stmt->fetch(PDO::FETCH_ASSOC);

// Fetch orders in driver area with user info
$ordersStmt = $pdo->prepare("SELECT o.*, u.name AS user_name, u.phone AS user_phone, u.address, u.gps_lat, u.gps_lng, u.profile_pic
                             FROM orders o 
                             JOIN users u ON o.user_id=u.id 
                             WHERE o.area_id=? 
                             ORDER BY o.created_at DESC");
$ordersStmt->execute([$driver['area_id']]);
$orders = $ordersStmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
  <meta charset="utf-8">
  <title><?php echo t('driver_panel'); ?> - <?php echo htmlspecialchars($driver['name']); ?></title>
  <link rel="stylesheet" href="assets/css/styles.css">
  <style>
    .profile-pic { width:40px; height:40px; border-radius:50%; object-fit:cover; margin-right:5px; vertical-align:middle; }
    .badge.pending { background:#f0ad4e; color:#fff; padding:2px 6px; border-radius:3px; }
    .badge.accepted { background:#5bc0de; color:#fff; padding:2px 6px; border-radius:3px; }
    .badge.delivered { background:#5cb85c; color:#fff; padding:2px 6px; border-radius:3px; }
    .notification {
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px;
        background: #5cb85c;
        color: white;
        border-radius: 5px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.2);
        z-index: 1000;
        display: none;
    }
    .connection-status {
        display: inline-block;
        padding: 4px 8px;
        border-radius: 4px;
        color: white;
        font-size: 12px;
        margin-right: 10px;
    }
    .connection-status.connected { background: #5cb85c; }
    .connection-status.disconnected { background: #d9534f; }
    .pulse {
        animation: pulse 2s infinite;
    }
    @keyframes pulse {
        0% { opacity: 1; }
        50% { opacity: 0.5; }
        100% { opacity: 1; }
    }
    .language-selector {
        display: inline-block;
        margin-right: 15px;
    }
    .language-selector select {
        padding: 5px;
        border-radius: 3px;
        border: 1px solid #ddd;
    }
  </style>
</head>
<body>
  <audio id="orderSound" preload="auto">
    <source src="order.mp3" type="audio/mpeg">
  </audio>
  
  <div class="notification" id="notification"><?php echo t('new_order_available'); ?></div>
  
  <div class="header">
    <div class="header-inner container">
      <div class="brand">
        <div class="logo"></div>
        <h1><?php echo t('driver_panel'); ?> — <?php echo htmlspecialchars($driver['name']); ?></h1>
      </div>
      <div>
        <div class="language-selector">
          <select id="languageSwitch">
            <option value="en" <?php echo $lang == 'en' ? 'selected' : ''; ?>><?php echo t('english'); ?></option>
            <option value="ar" <?php echo $lang == 'ar' ? 'selected' : ''; ?>><?php echo t('arabic'); ?></option>
            <option value="ku" <?php echo $lang == 'ku' ? 'selected' : ''; ?>><?php echo t('kurdish'); ?></option>
          </select>
        </div>
        <span id="connection-status" class="connection-status connected pulse"><?php echo t('live_updates'); ?></span>
        <a class="btn secondary" href="driver_login.php?logout=1"><?php echo t('logout'); ?></a>
      </div>
    </div>
  </div>

  <main class="container">
    <div class="kpis" id="kpis">
      <div class="kpi"><h3><?php echo t('main_stock'); ?></h3><div id="stock-value"><?php echo (int)$driver['stock']; ?></div></div>
      <div class="kpi"><h3><?php echo t('empty'); ?></h3><div id="empty-stock-value"><?php echo (int)$driver['empty_stock']; ?></div></div>
      <div class="kpi"><h3><?php echo t('sold'); ?></h3><div id="sold-value"><?php echo (int)$driver['sold']; ?></div></div>
      <div class="kpi"><h3><?php echo t('orders'); ?></h3><div id="orders-count"><?php echo count($orders); ?></div></div>
      <div class="kpi"><h3><?php echo t('connection_status'); ?></h3><div id="refresh-status"><?php echo t('live_updates'); ?></div></div>
    </div>

    <div class="card" style="margin-top:16px;">
      <h1><?php echo t('orders_in_zone'); ?> <span id="orders-badge" class="badge"><?php echo count($orders); ?></span></h1>
      <div class="hr"></div>
      <div id="orders-table-container">
        <table class="table" id="orders-table">
          <thead>
            <tr>
              <th>ID</th>
              <th><?php echo t('user'); ?></th>
              <th><?php echo t('phone'); ?></th>
              <th><?php echo t('address'); ?></th>
              <th><?php echo t('status'); ?></th>
              <th><?php echo t('gps'); ?></th>
              <th><?php echo t('actions'); ?></th>
            </tr>
          </thead>
          <tbody id="orders-tbody">
          <?php foreach($orders as $o): 
              $pic = !empty($o['profile_pic']) ? "uploads/profile_pics/" . htmlspecialchars($o['profile_pic']) : "assets/images/default-user.png";
          ?>
            <tr id="order-<?php echo $o['id']; ?>">
              <td>#<?php echo $o['id']; ?></td>
              <td><img class="profile-pic" src="<?php echo $pic; ?>" alt="Profile"><?php echo htmlspecialchars($o['user_name']); ?></td>
              <td><?php echo htmlspecialchars($o['user_phone']); ?></td>
              <td><?php echo htmlspecialchars($o['address'] ?? '-'); ?></td>
              <td><span class="badge <?php echo $o['status']; ?>" id="status-<?php echo $o['id']; ?>"><?php echo t($o['status']); ?></span></td>
              <td>
                <?php if($o['gps_lat'] && $o['gps_lng']): ?>
                  <a class="btn secondary" target="_blank"
                     href="https://www.google.com/maps?q=<?php echo $o['gps_lat']; ?>,<?php echo $o['gps_lng']; ?>"><?php echo t('map'); ?></a>
                <?php else: ?><span class="muted">—</span><?php endif; ?>
              </td>
              <td id="actions-<?php echo $o['id']; ?>">
                <?php if ($o['status']=='pending'): ?>
                  <a class="btn take-order" href="#" data-order-id="<?php echo $o['id']; ?>"><?php echo t('take'); ?></a>
                <?php elseif ($o['status']=='accepted' && $o['driver_id']==$driver_id): ?>
                  <a class="btn warn mark-delivered" href="#" data-order-id="<?php echo $o['id']; ?>"><?php echo t('mark_delivered'); ?></a>
                <?php else: ?>
                  <span class="muted">—</span>
                <?php endif; ?>
              </td>
            </tr>
          <?php endforeach; ?>
          </tbody>
        </table>
      </div>
    </div>

    <div class="card" style="margin-top:16px;">
      <h2><?php echo t('manual_stock_management'); ?></h2>
      <div class="hr"></div>
      <form method="post" id="stock-form">
        <div class="grid grid-2">
          <div>
            <label><?php echo t('exchange_empty_cylinders'); ?></label>
            <input type="number" class="input" name="exchange" id="exchange-input" min="0" max="<?php echo $driver['filled_stock']; ?>" placeholder="<?php echo t('sold_cylinders_returned'); ?>">
          </div>
          <div>
            <label><?php echo t('manual_sell'); ?></label>
            <input type="number" class="input" name="manual_sell" id="manual-sell-input" min="0" max="<?php echo $driver['filled_stock']; ?>" placeholder="<?php echo t('sell_filled_cylinders'); ?>">
          </div>
          <div style="grid-column:1 / -1">
            <button type="submit" class="btn"><?php echo t('update_stock'); ?></button>
          </div>
        </div>
      </form>
    </div>

  </main>

  <div class="footer">© <?php echo date('Y'); ?> <?php echo t('propane_service'); ?></div>

  <script>
    let lastUpdateTime = Date.now();
    let isPolling = true;
    let currentOrderCount = <?php echo count($orders); ?>;
    let audioPlayed = false;
    
    function playOrderSound() {
        const audio = document.getElementById('orderSound');
        if (audio) {
            audio.currentTime = 0;
            audio.play().catch(e => {
                console.log('Audio play failed:', e);
            });
        }
    }
    
    function showNotification(message) {
      const notification = document.getElementById('notification');
      notification.textContent = message;
      notification.style.display = 'block';
      
      setTimeout(() => {
        notification.style.display = 'none';
      }, 5000);
    }
    
    function updateUI(data) {
      // Update driver stock information
      document.getElementById('stock-value').textContent = data.driver.stock || 0;
      document.getElementById('empty-stock-value').textContent = data.driver.empty_stock || 0;
      document.getElementById('sold-value').textContent = data.driver.sold || 0;
      
      // Check if there are new orders
      if (data.orders.length > currentOrderCount) {
        showNotification('<?php echo t('new_order_available_in_zone'); ?>');
        playOrderSound();
      }
      
      // Update current order count
      currentOrderCount = data.orders.length;
      
      // Update orders count
      document.getElementById('orders-count').textContent = data.orders.length;
      document.getElementById('orders-badge').textContent = data.orders.length;
      
      // Update orders table
      updateOrdersTable(data.orders);
    }
    
    function updateOrdersTable(orders) {
      const tbody = document.getElementById('orders-tbody');
      
      // Clear the table
      tbody.innerHTML = '';
      
      // Add all orders
      orders.forEach(order => {
        const pic = order.profile_pic ? 
          `uploads/profile_pics/${order.profile_pic}` : 
          'assets/images/default-user.png';
        
        let actionHtml = '';
        if (order.status === 'pending') {
          actionHtml = `<a class="btn take-order" href="#" data-order-id="${order.id}"><?php echo t('take'); ?></a>`;
        } else if (order.status === 'accepted' && order.driver_id == <?php echo $driver_id; ?>) {
          actionHtml = `<a class="btn warn mark-delivered" href="#" data-order-id="${order.id}"><?php echo t('mark_delivered'); ?></a>`;
        } else {
          actionHtml = '<span class="muted">—</span>';
        }
        
        const gpsHtml = order.gps_lat && order.gps_lng ? 
          `<a class="btn secondary" target="_blank" href="https://www.google.com/maps?q=${order.gps_lat},${order.gps_lng}"><?php echo t('map'); ?></a>` :
          '<span class="muted">—</span>';
        
        const row = document.createElement('tr');
        row.id = `order-${order.id}`;
        row.innerHTML = `
          <td>#${order.id}</td>
          <td><img class="profile-pic" src="${pic}" alt="Profile">${escapeHtml(order.user_name)}</td>
          <td>${escapeHtml(order.user_phone)}</td>
          <td>${escapeHtml(order.address || '-')}</td>
          <td><span class="badge ${order.status}" id="status-${order.id}"><?php echo '${t(order.status)}'; ?></span></td>
          <td>${gpsHtml}</td>
          <td id="actions-${order.id}">${actionHtml}</td>
        `;
        
        tbody.appendChild(row);
      });
      
      // Reattach event listeners
      attachEventListeners();
    }
    
    function attachEventListeners() {
      // Attach click handlers to Take buttons
      document.querySelectorAll('.take-order').forEach(button => {
        button.addEventListener('click', function(e) {
          e.preventDefault();
          const orderId = this.getAttribute('data-order-id');
          takeOrder(orderId);
        });
      });
      
      // Attach click handlers to Mark Delivered buttons
      document.querySelectorAll('.mark-delivered').forEach(button => {
        button.addEventListener('click', function(e) {
          e.preventDefault();
          const orderId = this.getAttribute('data-order-id');
          markOrderDelivered(orderId);
        });
      });
    }
    
    function takeOrder(orderId) {
      fetch(`?ajax=take_order&order_id=${orderId}`)
        .then(response => response.json())
        .then(result => {
          if (result.success) {
            // Refresh the data
            checkForUpdates();
          } else {
            alert('Failed to take order: ' + result.message);
          }
        })
        .catch(error => {
          console.error('Error taking order:', error);
          alert('Error taking order');
        });
    }
    
    function markOrderDelivered(orderId) {
      fetch(`?ajax=mark_delivered&order_id=${orderId}`)
        .then(response => response.json())
        .then(result => {
          if (result.success) {
            // Refresh the data
            checkForUpdates();
          } else {
            alert('Failed to mark order as delivered: ' + result.message);
          }
        })
        .catch(error => {
          console.error('Error marking order as delivered:', error);
          alert('Error marking order as delivered');
        });
    }
    
    function checkForUpdates() {
      if (!isPolling) return;
      
      fetch(`?ajax=get_updates&t=${Date.now()}`)
        .then(response => {
          if (!response.ok) {
            throw new Error('Network response was not ok');
          }
          return response.json();
        })
        .then(data => {
          updateUI(data);
          lastUpdateTime = Date.now();
          updateConnectionStatus(true);
        })
        .catch(error => {
          console.error('Error fetching updates:', error);
          updateConnectionStatus(false);
        })
        .finally(() => {
          // Schedule next update
          setTimeout(checkForUpdates, 3000);
        });
    }
    
    function updateConnectionStatus(connected) {
      const statusElement = document.getElementById('connection-status');
      if (connected) {
        statusElement.textContent = '<?php echo t('live_updates'); ?>';
        statusElement.className = 'connection-status connected pulse';
      } else {
        statusElement.textContent = '<?php echo t('disconnected'); ?>';
        statusElement.className = 'connection-status disconnected';
        statusElement.classList.remove('pulse');
      }
    }
    
    function escapeHtml(unsafe) {
      return unsafe
        .replace(/&/g, "&amp;")
        .replace(/</g, "&lt;")
        .replace(/>/g, "&gt;")
        .replace(/"/g, "&quot;")
        .replace(/'/g, "&#039;");
    }
    
    // Translation helper for JavaScript
    const translations = {
        'en': <?php echo json_encode($texts['en']); ?>,
        'ar': <?php echo json_encode($texts['ar']); ?>,
        'ku': <?php echo json_encode($texts['ku']); ?>
    };
    
    function t(key) {
        const lang = document.documentElement.lang;
        return translations[lang] && translations[lang][key] ? translations[lang][key] : key;
    }
    
    // Initialize when page loads
    document.addEventListener('DOMContentLoaded', function() {
      // Start polling for updates
      checkForUpdates();
      
      // Attach event listeners to existing buttons
      attachEventListeners();
      
      // Handle stock form submission with AJAX
      document.getElementById('stock-form').addEventListener('submit', function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        
        fetch('?ajax=update_stock', {
          method: 'POST',
          body: formData
        })
        .then(response => response.json())
        .then(result => {
          if (result.success) {
            // Refresh the data to get updated stock values
            checkForUpdates();
            // Reset form
            document.getElementById('exchange-input').value = '';
            document.getElementById('manual-sell-input').value = '';
          } else {
            alert('Failed to update stock: ' + result.message);
          }
        })
        .catch(error => {
          console.error('Error updating stock:', error);
          alert('Error updating stock');
        });
      });
      
      // Language switcher
      document.getElementById('languageSwitch').addEventListener('change', function() {
        window.location.href = '?lang=' + this.value;
      });
      
      // Stop polling when page is hidden (to save resources)
      document.addEventListener('visibilitychange', function() {
        if (document.hidden) {
          isPolling = false;
        } else {
          isPolling = true;
          checkForUpdates();
        }
      });
    });
  </script>
</body>
</html>