<?php
session_start();
require 'db.php';

// User must be logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

$user_id = $_SESSION['user_id'];
$order_id = intval($_GET['order_id'] ?? 0);

// Fetch order with driver info
$stmt = $pdo->prepare("
    SELECT o.id, o.status, o.driver_id, d.name AS driver_name 
    FROM orders o
    LEFT JOIN drivers d ON o.driver_id = d.id
    WHERE o.id=? AND o.user_id=?
");
$stmt->execute([$order_id, $user_id]);
$order = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$order) {
    die("Order not found.");
}
?>
<!DOCTYPE html>
<html>
<head>
  <meta charset="utf-8">
  <title>Track Order #<?php echo $order['id']; ?></title>
  <link rel="stylesheet" href="https://unpkg.com/leaflet/dist/leaflet.css"/>
  <style>
    #map { height: 400px; margin:20px 0; }
  </style>
</head>
<body>
  <h2>Order #<?php echo $order['id']; ?> (<?php echo htmlspecialchars($order['status']); ?>)</h2>
  <?php if ($order['driver_id']): ?>
    <p>Driver: <?php echo htmlspecialchars($order['driver_name']); ?></p>
    <div id="map"></div>
    <script src="https://unpkg.com/leaflet/dist/leaflet.js"></script>
    <script>
      var map = L.map('map').setView([36.19, 44.01], 13); // Default center
      L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
          attribution: '&copy; OpenStreetMap contributors'
      }).addTo(map);

      var driverMarker = L.marker([36.19, 44.01]).addTo(map).bindPopup("Driver");

      async function fetchDriverLocation() {
          try {
              let res = await fetch("get_driver_location.php?driver_id=<?php echo $order['driver_id']; ?>");
              let data = await res.json();
              if (data.lat && data.lng) {
                  driverMarker.setLatLng([data.lat, data.lng]);
                  map.setView([data.lat, data.lng], 14);
              }
          } catch(e) {
              console.error("Error fetching location", e);
          }
      }

      // Fetch every 10s
      fetchDriverLocation();
      setInterval(fetchDriverLocation, 10000);
    </script>
  <?php else: ?>
    <p>Driver not yet assigned.</p>
  <?php endif; ?>
</body>
</html>
