<?php
session_start();
require 'db.php';
require 'lang.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: create_profile.php");
    exit;
}

$user_id = $_SESSION['user_id'];

// Fetch current user info
$stmt = $pdo->prepare("SELECT * FROM users WHERE id=?");
$stmt->execute([$user_id]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);

// Handle update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Handle profile picture upload
    $profile_pic = $user['profile_pic']; // Keep existing picture by default
    
    if (isset($_FILES['profile_pic']) && $_FILES['profile_pic']['error'] === UPLOAD_ERR_OK) {
        $uploadResult = handleProfilePicUpload($_FILES['profile_pic']);
        if ($uploadResult['success']) {
            $profile_pic = $uploadResult['filename'];
            
            // Delete old profile picture if it exists and is not the default
            if ($user['profile_pic'] && $user['profile_pic'] != 'default-avatar.png' && file_exists('uploads/profile_pics/' . $user['profile_pic'])) {
                unlink('uploads/profile_pics/' . $user['profile_pic']);
            }
        } else {
            $upload_error = $uploadResult['error'];
        }
    }
    
    $stmt = $pdo->prepare("UPDATE users 
        SET name=?, phone=?, address=?, area_id=?, gps_lat=?, gps_lng=?, profile_pic=?
        WHERE id=?");
    $stmt->execute([
        $_POST['name'], 
        $_POST['phone'], 
        $_POST['address'], 
        $_POST['area_id'],
        $_POST['gps_lat'], 
        $_POST['gps_lng'], 
        $profile_pic,
        $user_id
    ]);
    header("Location: profile.php?updated=1");
    exit;
}

// Function to handle profile picture upload and compression
function handleProfilePicUpload($file) {
    $uploadDir = 'uploads/profile_pics/';
    
    // Create directory if it doesn't exist
    if (!file_exists($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    // Validate file type
    $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
    $fileType = mime_content_type($file['tmp_name']);
    
    if (!in_array($fileType, $allowedTypes)) {
        return ['success' => false, 'error' => 'Invalid file type. Only JPG, PNG, and GIF are allowed.'];
    }
    
    // Validate file size (max 5MB before compression)
    if ($file['size'] > 5 * 1024 * 1024) {
        return ['success' => false, 'error' => 'File too large. Maximum size is 5MB.'];
    }
    
    // Generate unique filename
    $fileExtension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = uniqid('profile_') . '_' . time() . '.jpg'; // Always save as JPG for better compression
    
    // Compress and resize image
    $compressionResult = compressImage($file['tmp_name'], $uploadDir . $filename, 10);
    
    if ($compressionResult['success']) {
        return ['success' => true, 'filename' => $filename, 'size' => $compressionResult['size']];
    } else {
        return ['success' => false, 'error' => $compressionResult['error']];
    }
}

// Function to compress image to approximately 10KB
function compressImage($sourcePath, $destinationPath, $targetSizeKB = 10) {
    try {
        // Get image info
        $imageInfo = getimagesize($sourcePath);
        $mimeType = $imageInfo['mime'];
        
        // Create image from source
        switch ($mimeType) {
            case 'image/jpeg':
                $image = imagecreatefromjpeg($sourcePath);
                break;
            case 'image/png':
                $image = imagecreatefrompng($sourcePath);
                break;
            case 'image/gif':
                $image = imagecreatefromgif($sourcePath);
                break;
            default:
                return ['success' => false, 'error' => 'Unsupported image format'];
        }
        
        if (!$image) {
            return ['success' => false, 'error' => 'Failed to create image resource'];
        }
        
        // Get original dimensions
        $originalWidth = imagesx($image);
        $originalHeight = imagesy($image);
        
        // Calculate new dimensions (max 300px on the longest side while maintaining aspect ratio)
        $maxDimension = 300;
        if ($originalWidth > $originalHeight) {
            $newWidth = $maxDimension;
            $newHeight = intval($originalHeight * $maxDimension / $originalWidth);
        } else {
            $newHeight = $maxDimension;
            $newWidth = intval($originalWidth * $maxDimension / $originalHeight);
        }
        
        // Create new image
        $newImage = imagecreatetruecolor($newWidth, $newHeight);
        
        // Preserve transparency for PNG and GIF
        if ($mimeType == 'image/png' || $mimeType == 'image/gif') {
            imagecolortransparent($newImage, imagecolorallocatealpha($newImage, 0, 0, 0, 127));
            imagealphablending($newImage, false);
            imagesavealpha($newImage, true);
        }
        
        // Resize image
        imagecopyresampled($newImage, $image, 0, 0, 0, 0, $newWidth, $newHeight, $originalWidth, $originalHeight);
        
        // Compress with quality adjustment to achieve ~10KB
        $quality = 85; // Start with good quality
        $maxAttempts = 10;
        $attempt = 0;
        
        do {
            // Output image to temporary buffer to check size
            ob_start();
            imagejpeg($newImage, null, $quality);
            $imageData = ob_get_clean();
            $currentSize = strlen($imageData);
            
            // Adjust quality based on current size
            if ($currentSize > $targetSizeKB * 1024) {
                $quality -= 10;
            } else {
                break;
            }
            
            $attempt++;
        } while ($quality > 10 && $attempt < $maxAttempts);
        
        // If still too large, resize to smaller dimensions
        if ($currentSize > $targetSizeKB * 1024 && $maxDimension > 150) {
            imagedestroy($newImage);
            imagedestroy($image);
            
            // Try again with smaller dimensions
            return compressImage($sourcePath, $destinationPath, $targetSizeKB);
        }
        
        // Save final image
        $success = imagejpeg($newImage, $destinationPath, max($quality, 10));
        $finalSize = filesize($destinationPath);
        
        // Clean up
        imagedestroy($newImage);
        imagedestroy($image);
        
        if ($success) {
            return [
                'success' => true, 
                'size' => round($finalSize / 1024, 2) . 'KB',
                'quality' => $quality,
                'dimensions' => $newWidth . 'x' . $newHeight
            ];
        } else {
            return ['success' => false, 'error' => 'Failed to save compressed image'];
        }
        
    } catch (Exception $e) {
        return ['success' => false, 'error' => 'Compression error: ' . $e->getMessage()];
    }
}

// Load areas
$areas = $pdo->query("SELECT * FROM areas")->fetchAll(PDO::FETCH_ASSOC);

// Set direction based on language
$dir = ($lang == 'ar') ? 'rtl' : 'ltr';
?>
<!DOCTYPE html>
<html dir="<?php echo $dir; ?>">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title><?php echo t('edit_profile'); ?> - Propane Service</title>
  <style>
    :root {
      --primary: #4cc9f0;
      --secondary: #4895ef;
      --accent: #4361ee;
      --success: #4ade80;
      --danger: #f75555;
      --warning: #f59e0b;
      --light: #ffffff;
      --dark: #1e293b;
      --gray: #64748b;
      --light-gray: #f1f5f9;
      --border-radius: 12px;
      --box-shadow: 0 10px 25px rgba(0,0,0,0.08);
      --transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
      --gradient: linear-gradient(135deg, #4cc9f0 0%, #4895ef 50%, #4361ee 100%);
    }

    * {
      margin: 0;
      padding: 0;
      box-sizing: border-box;
      font-family: 'Segoe UI', system-ui, -apple-system, sans-serif;
    }

    body {
      background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
      min-height: 100vh;
      padding: 0;
      color: #334155;
      direction: <?php echo $dir; ?>;
      line-height: 1.6;
    }

    /* Modern Compact Header */
    .header {
      background: var(--light);
      box-shadow: 0 4px 20px rgba(76, 201, 240, 0.15);
      padding: 8px 5%;
      display: flex;
      justify-content: space-between;
      align-items: center;
      flex-wrap: wrap;
      gap: 8px;
      width: 100%;
      position: sticky;
      top: 0;
      z-index: 1000;
      min-height: 52px;
      border-bottom: 1px solid rgba(76, 201, 240, 0.2);
      backdrop-filter: blur(10px);
      animation: slideDown 0.5s ease-out;
    }

    @keyframes slideDown {
      from {
        transform: translateY(-100%);
        opacity: 0;
      }
      to {
        transform: translateY(0);
        opacity: 1;
      }
    }

    .brand {
      display: flex;
      align-items: center;
      gap: 10px;
      flex-shrink: 0;
    }

    .logo {
      width: 28px;
      height: 28px;
      background: var(--gradient);
      border-radius: 8px;
      display: flex;
      align-items: center;
      justify-content: center;
      color: white;
      font-weight: bold;
      flex-shrink: 0;
      font-size: 0.8rem;
      box-shadow: 0 4px 12px rgba(76, 201, 240, 0.3);
      animation: pulse 2s infinite;
    }

    @keyframes pulse {
      0%, 100% { transform: scale(1); }
      50% { transform: scale(1.05); }
    }

    h1, h2 {
      color: var(--accent);
      word-wrap: break-word;
    }

    h1 {
      font-size: 1.2rem;
      margin: 0;
      font-weight: 700;
      background: var(--gradient);
      -webkit-background-clip: text;
      -webkit-text-fill-color: transparent;
      background-clip: text;
    }

    h2 {
      font-size: 1.3rem;
      font-weight: 600;
    }

    .language-switcher {
      display: flex;
      gap: 4px;
      flex-wrap: wrap;
      background: var(--light-gray);
      padding: 2px;
      border-radius: 8px;
    }

    .lang-btn {
      padding: 4px 8px;
      border: none;
      border-radius: 6px;
      background: transparent;
      color: var(--gray);
      cursor: pointer;
      transition: var(--transition);
      text-decoration: none;
      font-size: 0.75rem;
      flex-shrink: 0;
      font-weight: 500;
    }

    .lang-btn.active {
      background: var(--primary);
      color: white;
      box-shadow: 0 2px 8px rgba(76, 201, 240, 0.3);
    }

    .lang-btn:hover {
      background: var(--primary);
      color: white;
      transform: translateY(-1px);
    }

    .btn {
      display: inline-flex;
      align-items: center;
      justify-content: center;
      padding: 6px 12px;
      background: var(--gradient);
      color: white;
      border: none;
      border-radius: 8px;
      cursor: pointer;
      text-decoration: none;
      font-weight: 600;
      transition: var(--transition);
      box-shadow: 0 4px 12px rgba(76, 201, 240, 0.3);
      text-align: center;
      white-space: nowrap;
      min-height: 32px;
      font-size: 0.8rem;
      position: relative;
      overflow: hidden;
    }

    .btn::before {
      content: '';
      position: absolute;
      top: 0;
      left: -100%;
      width: 100%;
      height: 100%;
      background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
      transition: left 0.5s;
    }

    .btn:hover::before {
      left: 100%;
    }

    .btn:hover {
      transform: translateY(-2px);
      box-shadow: 0 6px 20px rgba(76, 201, 240, 0.4);
    }

    .btn.secondary {
      background: var(--light);
      color: var(--primary);
      border: 1.5px solid var(--primary);
      box-shadow: none;
    }

    .btn.secondary:hover {
      background: var(--primary);
      color: white;
      transform: translateY(-2px);
      box-shadow: 0 4px 12px rgba(76, 201, 240, 0.3);
    }

    .container {
      width: 100%;
      max-width: 100%;
      margin: 0 auto;
      padding: 20px 5%;
    }

    .card {
      background: var(--light);
      border-radius: var(--border-radius);
      box-shadow: var(--box-shadow);
      padding: clamp(20px, 4vw, 30px);
      margin-bottom: 25px;
      overflow: hidden;
      border: 1px solid rgba(76, 201, 240, 0.1);
      animation: fadeInUp 0.6s ease-out;
      position: relative;
    }

    @keyframes fadeInUp {
      from {
        opacity: 0;
        transform: translateY(30px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    .card::before {
      content: '';
      position: absolute;
      top: 0;
      left: 0;
      right: 0;
      height: 2px;
      background: var(--gradient);
      transform: scaleX(0);
      transition: transform 0.3s ease;
    }

    .card:hover::before {
      transform: scaleX(1);
    }

    .card:hover {
      transform: translateY(-5px);
      box-shadow: 0 15px 30px rgba(76, 201, 240, 0.15);
    }

    .hr {
      height: 1px;
      background: linear-gradient(90deg, transparent, var(--primary), transparent);
      margin: 20px 0;
    }

    .muted {
      color: var(--gray);
      line-height: 1.6;
      font-size: 0.9rem;
    }

    .success {
      background: rgba(74, 222, 128, 0.1);
      color: #16a34a;
      padding: 12px 16px;
      border-radius: var(--border-radius);
      margin-bottom: 20px;
      border-left: 4px solid var(--success);
      animation: slideInRight 0.5s ease-out;
    }

    .error {
      background: rgba(247, 85, 85, 0.1);
      color: #dc2626;
      padding: 12px 16px;
      border-radius: var(--border-radius);
      margin-bottom: 20px;
      border-left: 4px solid var(--danger);
      animation: slideInRight 0.5s ease-out;
    }

    @keyframes slideInRight {
      from {
        opacity: 0;
        transform: translateX(30px);
      }
      to {
        opacity: 1;
        transform: translateX(0);
      }
    }

    .footer {
      text-align: center;
      padding: 20px;
      color: var(--gray);
      margin-top: 40px;
      background: var(--light);
      border-top: 1px solid rgba(76, 201, 240, 0.2);
      font-size: clamp(0.9rem, 2vw, 1rem);
    }

    /* Header actions - compact */
    .header-actions {
      display: flex;
      align-items: center;
      gap: 8px;
      flex-wrap: wrap;
    }

    .nav-buttons {
      display: flex;
      gap: 6px;
      flex-wrap: wrap;
    }

    /* Form Styles */
    .grid {
      display: grid;
      gap: 20px;
      margin-top: 20px;
    }

    .grid-2 {
      grid-template-columns: repeat(auto-fit, minmax(min(100%, 300px), 1fr));
    }

    label {
      display: block;
      margin-bottom: 8px;
      font-weight: 600;
      color: var(--dark);
      font-size: 0.9rem;
    }

    .input, select {
      width: 100%;
      padding: 12px 15px;
      border: 1px solid rgba(76, 201, 240, 0.2);
      border-radius: 8px;
      font-size: 0.9rem;
      transition: var(--transition);
      background: var(--light);
      color: var(--dark);
    }

    .input:focus, select:focus {
      outline: none;
      border-color: var(--primary);
      box-shadow: 0 0 0 3px rgba(76, 201, 240, 0.2);
      transform: translateY(-1px);
    }

    /* Profile Picture Section */
    .profile-pic-section {
      grid-column: 1 / -1;
      border: 2px dashed rgba(76, 201, 240, 0.3);
      border-radius: var(--border-radius);
      padding: 25px;
      text-align: center;
      background: var(--light-gray);
      transition: var(--transition);
    }

    .profile-pic-section:hover {
      border-color: var(--primary);
      background: rgba(76, 201, 240, 0.05);
    }

    .profile-pic-preview {
      width: 120px;
      height: 120px;
      border-radius: 50%;
      object-fit: cover;
      border: 3px solid var(--primary);
      margin: 0 auto 15px;
      display: block;
      box-shadow: 0 8px 20px rgba(76, 201, 240, 0.3);
      transition: var(--transition);
    }

    .profile-pic-preview:hover {
      transform: scale(1.05);
    }

    .profile-pic-upload {
      display: flex;
      flex-direction: column;
      align-items: center;
      gap: 10px;
    }

    .file-input-wrapper {
      position: relative;
      display: inline-block;
    }

    .file-input {
      position: absolute;
      left: 0;
      top: 0;
      opacity: 0;
      width: 100%;
      height: 100%;
      cursor: pointer;
    }

    .file-input-label {
      display: inline-block;
      padding: 8px 16px;
      background: var(--light);
      color: var(--primary);
      border: 1.5px solid var(--primary);
      border-radius: 8px;
      cursor: pointer;
      transition: var(--transition);
      font-weight: 500;
      font-size: 0.8rem;
    }

    .file-input-label:hover {
      background: var(--primary);
      color: white;
      transform: translateY(-1px);
    }

    .gps-section {
      grid-column: 1 / -1;
      display: flex;
      gap: 12px;
      align-items: center;
      flex-wrap: wrap;
      padding: 15px;
      background: rgba(76, 201, 240, 0.05);
      border-radius: var(--border-radius);
      border: 1px solid rgba(76, 201, 240, 0.2);
    }

    .btn.block {
      display: block;
      width: 100%;
      text-align: center;
      padding: 12px;
      font-size: 0.9rem;
    }

    /* Responsive adjustments */
    @media (max-width: 768px) {
      .header {
        padding: 6px 15px;
        gap: 6px;
      }
      
      .brand {
        gap: 8px;
      }
      
      .logo {
        width: 24px;
        height: 24px;
        font-size: 0.7rem;
      }
      
      h1 {
        font-size: 1.1rem;
      }
      
      .header-actions {
        gap: 6px;
      }
      
      .nav-buttons {
        gap: 4px;
      }
      
      .btn {
        padding: 5px 10px;
        font-size: 0.75rem;
        min-height: 28px;
      }
      
      .lang-btn {
        padding: 3px 6px;
        font-size: 0.7rem;
      }
      
      .grid-2 {
        grid-template-columns: 1fr;
      }
      
      .gps-section {
        flex-direction: column;
        align-items: stretch;
      }
    }

    @media (max-width: 480px) {
      .header {
        padding: 6px 10px;
        min-height: 48px;
      }
      
      .brand {
        flex-direction: row;
        gap: 6px;
      }
      
      .logo {
        width: 22px;
        height: 22px;
        font-size: 0.65rem;
      }
      
      h1 {
        font-size: 1rem;
      }
      
      .header-actions {
        flex-direction: column;
        align-items: flex-end;
        gap: 4px;
      }
      
      .language-switcher {
        order: -1;
        justify-content: flex-end;
      }
      
      .nav-buttons {
        justify-content: flex-end;
      }
      
      .btn {
        padding: 4px 8px;
        font-size: 0.7rem;
        min-height: 26px;
      }
      
      .profile-pic-preview {
        width: 100px;
        height: 100px;
      }
    }

    @media (max-width: 360px) {
      .header {
        padding: 4px 8px;
      }
      
      .brand {
        gap: 4px;
      }
      
      h1 {
        font-size: 0.9rem;
      }
      
      .nav-buttons {
        flex-direction: column;
        gap: 3px;
        width: 100%;
      }
      
      .nav-buttons .btn {
        width: auto;
        min-width: 100px;
      }
    }

    /* Extra small devices */
    @media (max-width: 320px) {
      .header {
        flex-direction: column;
        align-items: stretch;
        gap: 6px;
        padding: 6px;
      }
      
      .brand {
        justify-content: center;
        text-align: center;
      }
      
      .header-actions {
        align-items: center;
        gap: 6px;
      }
      
      .language-switcher {
        justify-content: center;
        width: 100%;
      }
      
      .nav-buttons {
        justify-content: center;
        width: 100%;
      }
      
      .nav-buttons .btn {
        width: 100%;
        max-width: 180px;
      }
    }
  </style>
  <script>
    document.addEventListener('DOMContentLoaded', function() {
      // Add animation to form elements
      const inputs = document.querySelectorAll('.input, select');
      inputs.forEach((input, index) => {
        input.style.animation = `fadeInUp 0.5s ease ${index * 0.1}s forwards`;
        input.style.opacity = 0;
      });

      // Success message animation
      const successMsg = document.querySelector('.success');
      if (successMsg) {
        successMsg.style.animation = 'slideInRight 0.5s ease';
      }

      // Profile picture preview
      const profilePicInput = document.getElementById('profile_pic');
      const profilePicPreview = document.getElementById('profile_preview');
      
      profilePicInput.addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (file) {
          const reader = new FileReader();
          reader.onload = function(e) {
            profilePicPreview.src = e.target.result;
            profilePicPreview.style.transform = 'scale(1.1)';
            setTimeout(() => {
              profilePicPreview.style.transform = 'scale(1)';
            }, 300);
          };
          reader.readAsDataURL(file);
          
          // Show file info
          const fileInfo = document.getElementById('file_info');
          fileInfo.textContent = `Selected: ${file.name} (${(file.size / 1024).toFixed(2)}KB)`;
          fileInfo.style.color = 'var(--primary)';
          fileInfo.style.fontWeight = '600';
        }
      });
    });

    function getGPSInto(latField, lngField, statusField) {
        const statusEl = document.querySelector(statusField);
        if (!navigator.geolocation) {
            statusEl.textContent = "Geolocation not supported";
            statusEl.style.color = 'var(--danger)';
            return;
        }
        statusEl.textContent = "Fetching location...";
        statusEl.style.color = 'var(--warning)';
        
        navigator.geolocation.getCurrentPosition(
            (pos) => {
                document.querySelector(latField).value = pos.coords.latitude;
                document.querySelector(lngField).value = pos.coords.longitude;
                statusEl.textContent = "✅ Location set successfully!";
                statusEl.style.color = 'var(--success)';
                
                // Add success animation
                setTimeout(() => {
                  statusEl.style.opacity = 0;
                  setTimeout(() => statusEl.textContent = "", 600);
                }, 2000);
            },
            (err) => {
                statusEl.textContent = "❌ Unable to fetch location";
                statusEl.style.color = 'var(--danger)';
            }
        );
    }
  </script>
</head>
<body>

<!-- Modern Compact Header -->
<div class="header">
  <div class="brand">
    <div class="logo">PS</div>
    <h1><?php echo t('edit_profile'); ?></h1>
  </div>
  
  <div class="header-actions">
    <div class="language-switcher">
      <a href="?lang=en" class="lang-btn <?php echo $lang == 'en' ? 'active' : ''; ?>">EN</a>
      <a href="?lang=ar" class="lang-btn <?php echo $lang == 'ar' ? 'active' : ''; ?>">AR</a>
      <a href="?lang=ku" class="lang-btn <?php echo $lang == 'ku' ? 'active' : ''; ?>">KU</a>
    </div>
    
    <div class="nav-buttons">
      <a class="btn secondary" href="user_dashboard.php"><?php echo t('dashboard_link'); ?></a>
      <a class="btn secondary" href="user_orders.php"><?php echo t('orders'); ?></a>
      <a class="btn secondary" href="logout.php"><?php echo t('logout'); ?></a>
    </div>
  </div>
</div>

<main class="container">
  <div class="card">
    <h2><?php echo t('edit_profile'); ?></h2>
    <?php if(isset($_GET['updated'])): ?>
      <p class="success">✅ <?php echo t('profile_updated'); ?></p>
    <?php endif; ?>
    
    <?php if(isset($upload_error)): ?>
      <p class="error">❌ <?php echo htmlspecialchars($upload_error); ?></p>
    <?php endif; ?>
    
    <div class="hr"></div>
    
    <form method="post" enctype="multipart/form-data" class="grid grid-2">
      <!-- Profile Picture Section -->
      <div class="profile-pic-section">
        <img id="profile_preview" src="uploads/profile_pics/<?php echo htmlspecialchars($user['profile_pic'] ?: 'default-avatar.png'); ?>" 
             alt="Profile Picture" class="profile-pic-preview"
             onerror="this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMTIwIiBoZWlnaHQ9IjEyMCIgdmlld0JveD0iMCAwIDEyMCAxMjAiIGZpbGw9Im5vbmUiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+PGNpcmNsZSBjeD0iNjAiIGN5PSI2MCIgcj0iNjAiIGZpbGw9IiNlOWVjZWYiLz48cGF0aCBkPSJNNjAgNjBDNzIuMTU3IDYwIDkwIDQ3LjU2OSA5MCAzMEM5MCAxMi40MzEgNzIuMTU3IDAgNjAgMEM0Ny44NDMgMCAzMCAxMi40MzEgMzAgMzBDMzAgNDcuNTY5IDQ3Ljg0MyA2MCA2MCA2MFoiIGZpbGw9IiM2Yzc1N2QiLz48Y2lyY2xlIGN4PSI2MCIgY3k9IjQ1IiByPSIyMCIgZmlsbD0iIzZjNzU3ZCIvPjwvc3ZnPg=='">
        
        <div class="profile-pic-upload">
          <div class="file-input-wrapper">
            <input type="file" id="profile_pic" name="profile_pic" class="file-input" 
                   accept="image/jpeg,image/jpg,image/png,image/gif">
            <label for="profile_pic" class="file-input-label">
              <?php echo t('choose_photo'); ?>
            </label>
          </div>
          <span id="file_info" class="muted"><?php echo t('max_size_10kb'); ?></span>
        </div>
      </div>

      <div>
        <label><?php echo t('name'); ?></label>
        <input class="input" type="text" name="name" 
               value="<?php echo htmlspecialchars($user['name']); ?>" required>
      </div>
      <div>
        <label><?php echo t('phone'); ?></label>
        <input class="input" type="text" name="phone" 
               value="<?php echo htmlspecialchars($user['phone']); ?>">
      </div>
      <div class="grid-2" style="grid-column:1 / -1">
        <div>
          <label><?php echo t('address'); ?></label>
          <input class="input" type="text" name="address" 
                 value="<?php echo htmlspecialchars($user['address']); ?>">
        </div>
        <div>
          <label><?php echo t('area'); ?></label>
          <select class="input" name="area_id">
            <?php foreach($areas as $a): ?>
              <option value="<?php echo $a['id']; ?>" 
                <?php echo ($a['id']==$user['area_id'])?'selected':''; ?>>
                <?php echo htmlspecialchars($a['name']); ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>
      </div>

      <div>
        <label><?php echo t('gps_latitude'); ?></label>
        <input id="gps_lat" class="input" type="text" 
               name="gps_lat" value="<?php echo htmlspecialchars($user['gps_lat']); ?>">
      </div>
      <div>
        <label><?php echo t('gps_longitude'); ?></label>
        <input id="gps_lng" class="input" type="text" 
               name="gps_lng" value="<?php echo htmlspecialchars($user['gps_lng']); ?>">
      </div>

      <div class="gps-section">
        <button class="btn" type="button" 
                onclick="getGPSInto('#gps_lat','#gps_lng','#gps_status')">
          📍 <?php echo t('use_my_location'); ?>
        </button>
        <span id="gps_status" class="muted"></span>
      </div>

      <div style="grid-column:1 / -1">
        <button class="btn block" type="submit"><?php echo t('update_profile'); ?></button>
      </div>
    </form>
  </div>
</main>

<div class="footer">© <?php echo date('Y'); ?> <?php echo t('propane_service'); ?></div>
</body>
</html>