<?php
session_start();
require 'db.php';
require 'lang.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

$user_id = $_SESSION['user_id'];
$message = '';

// Cancel only pending orders
if (isset($_GET['cancel_id'])) {
    $cancel_id = (int)$_GET['cancel_id'];
    $stmt = $pdo->prepare("UPDATE orders SET status='cancelled' WHERE id=? AND user_id=? AND status='pending'");
    $stmt->execute([$cancel_id, $user_id]);
    if ($stmt->rowCount()) {
        $message = "✅ Order #$cancel_id has been cancelled!";
    } else {
        $message = "⚠️ Only pending orders can be cancelled.";
    }
}

// Fetch ALL orders including cancelled ones
$stmt = $pdo->prepare("
    SELECT o.*, a.name AS area_name, d.name AS driver_name, d.phone AS driver_phone, 
           d.gps_lat AS driver_lat, d.gps_lng AS driver_lng, d.last_location_update
    FROM orders o
    LEFT JOIN areas a ON o.area_id = a.id
    LEFT JOIN drivers d ON o.driver_id = d.id
    WHERE o.user_id = ?
    ORDER BY o.created_at DESC
");
$stmt->execute([$user_id]);
$orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Set direction based on language
$dir = ($lang == 'ar') ? 'rtl' : 'ltr';
?>
<!DOCTYPE html>
<html dir="<?php echo $dir; ?>" lang="<?php echo $lang; ?>">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title><?php echo t('orders'); ?> - Propane Service</title>
<style>
    :root {
      --primary: #4cc9f0;
      --secondary: #4895ef;
      --accent: #4361ee;
      --success: #4ade80;
      --danger: #f75555;
      --warning: #f59e0b;
      --light: #ffffff;
      --dark: #1e293b;
      --gray: #64748b;
      --light-gray: #f1f5f9;
      --border-radius: 12px;
      --box-shadow: 0 10px 25px rgba(0,0,0,0.08);
      --transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
      --gradient: linear-gradient(135deg, #4cc9f0 0%, #4895ef 50%, #4361ee 100%);
    }

    * {
      margin: 0;
      padding: 0;
      box-sizing: border-box;
      font-family: 'Segoe UI', system-ui, -apple-system, sans-serif;
    }

    body {
      background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
      min-height: 100vh;
      padding: 0;
      color: #334155;
      direction: <?php echo $dir; ?>;
      line-height: 1.6;
    }

    /* Modern Compact Header */
    .header {
      background: var(--light);
      box-shadow: 0 4px 20px rgba(76, 201, 240, 0.15);
      padding: 8px 5%;
      display: flex;
      justify-content: space-between;
      align-items: center;
      flex-wrap: wrap;
      gap: 8px;
      width: 100%;
      position: sticky;
      top: 0;
      z-index: 1000;
      min-height: 52px;
      border-bottom: 1px solid rgba(76, 201, 240, 0.2);
      backdrop-filter: blur(10px);
      animation: slideDown 0.5s ease-out;
    }

    @keyframes slideDown {
      from {
        transform: translateY(-100%);
        opacity: 0;
      }
      to {
        transform: translateY(0);
        opacity: 1;
      }
    }

    .brand {
      display: flex;
      align-items: center;
      gap: 10px;
      flex-shrink: 0;
    }

    .logo {
      width: 28px;
      height: 28px;
      background: var(--gradient);
      border-radius: 8px;
      display: flex;
      align-items: center;
      justify-content: center;
      color: white;
      font-weight: bold;
      flex-shrink: 0;
      font-size: 0.8rem;
      box-shadow: 0 4px 12px rgba(76, 201, 240, 0.3);
      animation: pulse 2s infinite;
    }

    @keyframes pulse {
      0%, 100% { transform: scale(1); }
      50% { transform: scale(1.05); }
    }

    h1, h2 {
      color: var(--accent);
      word-wrap: break-word;
    }

    h1 {
      font-size: 1.2rem;
      margin: 0;
      font-weight: 700;
      background: var(--gradient);
      -webkit-background-clip: text;
      -webkit-text-fill-color: transparent;
      background-clip: text;
    }

    h2 {
      font-size: 1.3rem;
      font-weight: 600;
    }

    .language-switcher {
      display: flex;
      gap: 4px;
      flex-wrap: wrap;
      background: var(--light-gray);
      padding: 2px;
      border-radius: 8px;
    }

    .lang-btn {
      padding: 4px 8px;
      border: none;
      border-radius: 6px;
      background: transparent;
      color: var(--gray);
      cursor: pointer;
      transition: var(--transition);
      text-decoration: none;
      font-size: 0.75rem;
      flex-shrink: 0;
      font-weight: 500;
    }

    .lang-btn.active {
      background: var(--primary);
      color: white;
      box-shadow: 0 2px 8px rgba(76, 201, 240, 0.3);
    }

    .lang-btn:hover {
      background: var(--primary);
      color: white;
      transform: translateY(-1px);
    }

    .btn {
      display: inline-flex;
      align-items: center;
      justify-content: center;
      padding: 6px 12px;
      background: var(--gradient);
      color: white;
      border: none;
      border-radius: 8px;
      cursor: pointer;
      text-decoration: none;
      font-weight: 600;
      transition: var(--transition);
      box-shadow: 0 4px 12px rgba(76, 201, 240, 0.3);
      text-align: center;
      white-space: nowrap;
      min-height: 32px;
      font-size: 0.8rem;
      position: relative;
      overflow: hidden;
    }

    .btn::before {
      content: '';
      position: absolute;
      top: 0;
      left: -100%;
      width: 100%;
      height: 100%;
      background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
      transition: left 0.5s;
    }

    .btn:hover::before {
      left: 100%;
    }

    .btn:hover {
      transform: translateY(-2px);
      box-shadow: 0 6px 20px rgba(76, 201, 240, 0.4);
    }

    .btn.secondary {
      background: var(--light);
      color: var(--primary);
      border: 1.5px solid var(--primary);
      box-shadow: none;
    }

    .btn.secondary:hover {
      background: var(--primary);
      color: white;
      transform: translateY(-2px);
      box-shadow: 0 4px 12px rgba(76, 201, 240, 0.3);
    }

    .container {
      width: 100%;
      max-width: 100%;
      margin: 0 auto;
      padding: 20px 5%;
    }

    .card {
      background: var(--light);
      border-radius: var(--border-radius);
      box-shadow: var(--box-shadow);
      padding: clamp(15px, 3vw, 25px);
      margin-bottom: 25px;
      overflow: hidden;
      border: 1px solid rgba(76, 201, 240, 0.1);
      animation: fadeInUp 0.6s ease-out;
    }

    @keyframes fadeInUp {
      from {
        opacity: 0;
        transform: translateY(30px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    .hr {
      height: 1px;
      background: linear-gradient(90deg, transparent, var(--primary), transparent);
      margin: 20px 0;
    }

    .new-order-msg {
      background: rgba(76, 201, 240, 0.1);
      padding: 12px 16px;
      margin-bottom: 20px;
      border-radius: var(--border-radius);
      border-left: 4px solid var(--primary);
      word-wrap: break-word;
      color: var(--dark);
      animation: slideInRight 0.5s ease-out;
    }

    @keyframes slideInRight {
      from {
        opacity: 0;
        transform: translateX(30px);
      }
      to {
        opacity: 1;
        transform: translateX(0);
      }
    }

    .badge {
      padding: 4px 10px;
      border-radius: 20px;
      font-size: 0.75rem;
      font-weight: 600;
      display: inline-block;
      text-align: center;
    }

    .pending {
      background: rgba(245, 158, 11, 0.1);
      color: #d97706;
    }

    .confirmed {
      background: rgba(76, 201, 240, 0.1);
      color: var(--primary);
    }

    .delivered {
      background: rgba(74, 222, 128, 0.1);
      color: #16a34a;
    }

    .cancelled {
      background: rgba(247, 85, 85, 0.1);
      color: #dc2626;
    }

    .cancel-btn {
      background: var(--danger);
      color: white;
      padding: 6px 12px;
      border: none;
      cursor: pointer;
      border-radius: 8px;
      font-size: 0.75rem;
      transition: var(--transition);
      text-decoration: none;
      display: inline-flex;
      align-items: center;
      justify-content: center;
      min-height: 28px;
      min-width: 70px;
      font-weight: 500;
    }

    .cancel-btn:hover {
      background: #dc2626;
      transform: scale(1.05);
      box-shadow: 0 4px 12px rgba(247, 85, 85, 0.3);
    }

    .footer {
      text-align: center;
      padding: 20px;
      color: var(--gray);
      margin-top: 40px;
      background: var(--light);
      border-top: 1px solid rgba(76, 201, 240, 0.2);
      font-size: clamp(0.9rem, 2vw, 1rem);
    }

    /* Header actions - compact */
    .header-actions {
      display: flex;
      align-items: center;
      gap: 8px;
      flex-wrap: wrap;
    }

    .nav-buttons {
      display: flex;
      gap: 6px;
      flex-wrap: wrap;
    }

    /* Orders grid */
    .orders-grid {
      display: grid;
      grid-template-columns: repeat(auto-fill, minmax(min(100%, 320px), 1fr));
      gap: 20px;
      margin-top: 20px;
    }

    .order-card {
      background: var(--light);
      border-radius: var(--border-radius);
      box-shadow: var(--box-shadow);
      padding: clamp(15px, 3vw, 20px);
      transition: var(--transition);
      border-left: 4px solid var(--primary);
      display: flex;
      flex-direction: column;
      height: 100%;
      position: relative;
      overflow: hidden;
    }

    .order-card::before {
      content: '';
      position: absolute;
      top: 0;
      left: 0;
      right: 0;
      height: 2px;
      background: var(--gradient);
      transform: scaleX(0);
      transition: transform 0.3s ease;
    }

    .order-card:hover::before {
      transform: scaleX(1);
    }

    .order-card.cancelled {
      border-left-color: var(--danger);
      opacity: 0.9;
    }

    .order-card:hover {
      transform: translateY(-5px);
      box-shadow: 0 15px 30px rgba(76, 201, 240, 0.15);
    }

    .order-header {
      display: flex;
      justify-content: space-between;
      align-items: flex-start;
      margin-bottom: 15px;
      padding-bottom: 10px;
      border-bottom: 1px solid rgba(76, 201, 240, 0.2);
      gap: 10px;
      flex-wrap: wrap;
    }

    .order-id {
      font-weight: bold;
      font-size: clamp(0.9rem, 3vw, 1rem);
      color: var(--accent);
      word-break: break-word;
    }

    .order-card.cancelled .order-id {
      color: var(--danger);
    }

    .order-date {
      color: var(--gray);
      font-size: clamp(0.75rem, 2.5vw, 0.8rem);
      white-space: nowrap;
    }

    .order-details {
      margin-bottom: 15px;
      flex-grow: 1;
    }

    .detail-row {
      display: flex;
      justify-content: space-between;
      margin-bottom: 8px;
      gap: 10px;
      flex-wrap: wrap;
    }

    .detail-label {
      font-weight: 600;
      color: var(--dark);
      font-size: clamp(0.8rem, 2.5vw, 0.9rem);
    }

    .detail-value {
      color: var(--gray);
      font-size: clamp(0.8rem, 2.5vw, 0.9rem);
      text-align: right;
      word-break: break-word;
    }

    .driver-info {
      background: rgba(76, 201, 240, 0.05);
      padding: 12px;
      border-radius: var(--border-radius);
      margin-top: 15px;
      border: 1px solid rgba(76, 201, 240, 0.1);
    }

    .driver-info-title {
      font-weight: 600;
      margin-bottom: 5px;
      color: var(--primary);
      font-size: clamp(0.8rem, 2.5vw, 0.9rem);
    }

    .driver-detail {
      display: flex;
      justify-content: space-between;
      gap: 10px;
      flex-wrap: wrap;
    }

    .order-actions {
      margin-top: 15px;
      text-align: center;
    }

    /* Animation */
    @keyframes fadeIn {
      from { opacity: 0; transform: translateY(20px); }
      to { opacity: 1; transform: translateY(0); }
    }

    .order-card {
      animation: fadeIn 0.5s ease forwards;
      opacity: 0;
    }

    /* Map Modal Styles */
    .modal {
        display: none;
        position: fixed;
        z-index: 2000;
        left: 0;
        top: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0,0,0,0.5);
        backdrop-filter: blur(5px);
    }
    
    .modal-content {
        position: relative;
        background-color: var(--light);
        margin: 2% auto;
        padding: 0;
        width: 90%;
        max-width: 800px;
        border-radius: 12px;
        box-shadow: 0 20px 40px rgba(0,0,0,0.2);
        animation: modalSlideIn 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        color: var(--dark);
        font-family: "Segoe UI", system-ui, sans-serif;
        border: 1px solid rgba(76, 201, 240, 0.2);
    }
    
    @keyframes modalSlideIn {
        from { transform: translateY(-50px) scale(0.9); opacity: 0; }
        to { transform: translateY(0) scale(1); opacity: 1; }
    }
    
    .modal-header {
        padding: 12px 20px;
        background: var(--gradient);
        border-bottom: 1px solid rgba(76, 201, 240, 0.2);
        border-radius: 12px 12px 0 0;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }
    
    .modal-header h3 {
        margin: 0;
        color: white;
        font-weight: 600;
    }
    
    .close-modal {
        background: rgba(255,255,255,0.2);
        border: none;
        font-size: 20px;
        cursor: pointer;
        color: white;
        padding: 0;
        width: 28px;
        height: 28px;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: 50%;
        transition: var(--transition);
    }
    
    .close-modal:hover {
        background: rgba(255,255,255,0.3);
        transform: rotate(90deg);
    }
    
    .modal-body {
        padding: 0;
    }
    
    #driver-map-container {
        width: 100%;
        height: 400px;
        border-radius: 0 0 12px 12px;
    }
    
    .map-info {
        padding: 15px 20px;
        background: var(--light-gray);
        border-top: 1px solid rgba(76, 201, 240, 0.1);
    }
    
    .map-info p {
        margin: 5px 0;
        font-size: 14px;
    }
    
    .map-actions {
        padding: 15px 20px;
        display: flex;
        gap: 10px;
        justify-content: flex-end;
        border-top: 1px solid rgba(76, 201, 240, 0.1);
    }
    
    .btn-small {
        padding: 6px 12px;
        font-size: 13px;
    }
    
    .fallback-map {
        background: var(--light-gray);
        display: flex;
        align-items: center;
        justify-content: center;
        height: 100%;
        color: var(--gray);
        font-size: 16px;
        flex-direction: column;
        gap: 10px;
    }
    
    .location-status {
        padding: 8px 12px;
        border-radius: 6px;
        font-size: 12px;
        margin: 5px 0;
        font-family: "Segoe UI", system-ui, sans-serif;
    }
    
    .location-status.success {
        background: #dcfce7;
        color: #166534;
        border: 1px solid #bbf7d0;
    }
    
    .location-status.error {
        background: #fee2e2;
        color: #991b1b;
        border: 1px solid #fecaca;
    }
    
    .location-status.warning {
        background: #fef3c7;
        color: #92400e;
        border: 1px solid #fde68a;
    }
    
    /* Ensure Leaflet map container has proper dimensions */
    .leaflet-container {
        height: 400px;
        width: 100%;
        border-radius: 0 0 12px 12px;
    }
    
    .distance-info {
        background: var(--light-gray);
        padding: 8px 12px;
        border-radius: 6px;
        margin: 5px 0;
        font-size: 13px;
        border-left: 4px solid var(--primary);
    }
    
    .track-driver-btn {
        background: var(--gradient);
        color: white;
        padding: 8px 16px;
        border: none;
        cursor: pointer;
        border-radius: 8px;
        font-size: 0.8rem;
        transition: var(--transition);
        text-decoration: none;
        display: inline-flex;
        align-items: center;
        justify-content: center;
        min-height: 32px;
        min-width: 110px;
        margin-top: 10px;
        font-weight: 600;
        box-shadow: 0 4px 12px rgba(76, 201, 240, 0.3);
    }
    
    .track-driver-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(76, 201, 240, 0.4);
    }
    
    .last-update {
        font-size: 0.75rem;
        color: var(--gray);
        margin-top: 5px;
    }

    /* Responsive adjustments for compact header */
    @media (max-width: 992px) {
      .header {
        padding: 8px 20px;
      }
      
      .container {
        padding: 20px;
      }
    }

    @media (max-width: 768px) {
      .header {
        padding: 6px 15px;
        gap: 6px;
      }
      
      .brand {
        gap: 8px;
      }
      
      .logo {
        width: 24px;
        height: 24px;
        font-size: 0.7rem;
      }
      
      h1 {
        font-size: 1.1rem;
      }
      
      .header-actions {
        gap: 6px;
      }
      
      .nav-buttons {
        gap: 4px;
      }
      
      .btn {
        padding: 5px 10px;
        font-size: 0.75rem;
        min-height: 28px;
      }
      
      .lang-btn {
        padding: 3px 6px;
        font-size: 0.7rem;
      }
      
      .map-actions {
        flex-wrap: wrap;
        justify-content: center;
      }
    }

    @media (max-width: 480px) {
      .header {
        padding: 6px 10px;
        min-height: 48px;
      }
      
      .brand {
        flex-direction: row;
        gap: 6px;
      }
      
      .logo {
        width: 22px;
        height: 22px;
        font-size: 0.65rem;
      }
      
      h1 {
        font-size: 1rem;
      }
      
      .header-actions {
        flex-direction: column;
        align-items: flex-end;
        gap: 4px;
      }
      
      .language-switcher {
        order: -1;
        justify-content: flex-end;
      }
      
      .nav-buttons {
        justify-content: flex-end;
      }
      
      .btn {
        padding: 4px 8px;
        font-size: 0.7rem;
        min-height: 26px;
      }
      
      .modal-content {
        width: 95%;
        margin: 5% auto;
      }
    }

    @media (max-width: 360px) {
      .header {
        padding: 4px 8px;
      }
      
      .brand {
        gap: 4px;
      }
      
      h1 {
        font-size: 0.9rem;
      }
      
      .nav-buttons {
        flex-direction: column;
        gap: 3px;
        width: 100%;
      }
      
      .nav-buttons .btn {
        width: auto;
        min-width: 100px;
      }
    }

    /* Extra small devices */
    @media (max-width: 320px) {
      .header {
        flex-direction: column;
        align-items: stretch;
        gap: 6px;
        padding: 6px;
      }
      
      .brand {
        justify-content: center;
        text-align: center;
      }
      
      .header-actions {
        align-items: center;
        gap: 6px;
      }
      
      .language-switcher {
        justify-content: center;
        width: 100%;
      }
      
      .nav-buttons {
        justify-content: center;
        width: 100%;
      }
      
      .nav-buttons .btn {
        width: 100%;
        max-width: 180px;
      }
    }
</style>
</head>
<body>
<!-- Driver Location Map Modal -->
<div id="driverMapModal" class="modal">
    <div class="modal-content">
      <div class="modal-header">
        <h3 id="driverModalTitle"><?php echo t('driver_location'); ?></h3>
        <button class="close-modal" id="closeDriverModal">&times;</button>
      </div>
      <div class="modal-body">
        <div id="driver-map-container">
          <div class="fallback-map" id="driverFallbackMap">
            <div>🌍 Loading driver location...</div>
            <div id="driverLocationStatus" class="location-status warning">
              Loading map...
            </div>
          </div>
        </div>
        <div class="map-info">
          <p><strong id="driverName"></strong></p>
          <p id="driverPhone"></p>
          <p id="driverLastUpdate"></p>
          <div id="driverDistanceInfo" class="distance-info" style="display: none;">
            📏 Distance: <span id="driverDistanceValue">Calculating...</span>
          </div>
        </div>
        <div class="map-actions">
          <button id="refreshDriverLocation" class="btn secondary btn-small">🔄 <?php echo t('refresh_location'); ?></button>
          <button id="closeDriverMap" class="btn secondary btn-small"><?php echo t('close'); ?></button>
        </div>
      </div>
    </div>
</div>

<!-- Modern Compact Header -->
<div class="header">
  <div class="brand">
    <div class="logo">PS</div>
    <h1><?php echo t('orders'); ?></h1>
  </div>
  
  <div class="header-actions">
    <div class="language-switcher">
      <a href="?lang=en" class="lang-btn <?php echo $lang == 'en' ? 'active' : ''; ?>">EN</a>
      <a href="?lang=ar" class="lang-btn <?php echo $lang == 'ar' ? 'active' : ''; ?>">AR</a>
      <a href="?lang=ku" class="lang-btn <?php echo $lang == 'ku' ? 'active' : ''; ?>">KU</a>
    </div>
    
    <div class="nav-buttons">
      <a class="btn secondary" href="user_dashboard.php"><?php echo t('dashboard_link'); ?></a>
      <a class="btn secondary" href="profile.php"><?php echo t('profile_link'); ?></a>
      <a class="btn secondary" href="logout.php"><?php echo t('logout'); ?></a>
    </div>
  </div>
</div>

<main class="container">
  <?php if($message): ?>
    <div class="new-order-msg"><?php echo $message; ?></div>
  <?php endif; ?>

  <div style="margin-bottom:20px; text-align: center;">
    <a href="user_dashboard.php" class="btn"><?php echo t('place_new_order'); ?></a>
  </div>

  <div class="card">
    <h2><?php echo t('previous_orders'); ?></h2>
    <div class="hr"></div>
    <?php if (empty($orders)): ?>
        <p style="text-align: center; padding: 20px; color: var(--gray);"><?php echo t('no_orders'); ?></p>
    <?php else: ?>
        <div class="orders-grid">
            <?php foreach ($orders as $o): 
                $hasDriverLocation = (!empty($o['driver_lat']) && !empty($o['driver_lng']) && 
                                     $o['status'] == 'accepted' && $o['driver_id']);
                $lastUpdate = $o['last_location_update'] ? 
                    date('Y-m-d H:i', strtotime($o['last_location_update'])) : 
                    'Unknown';
            ?>
            <div class="order-card <?php echo $o['status'] == 'cancelled' ? 'cancelled' : ''; ?>">
                <div class="order-header">
                    <div class="order-id"><?php echo t('order'); ?> #<?php echo $o['id']; ?></div>
                    <div class="order-date"><?php echo date('Y-m-d H:i', strtotime($o['created_at'])); ?></div>
                </div>
                
                <div class="order-details">
                    <div class="detail-row">
                        <span class="detail-label"><?php echo t('area'); ?>:</span>
                        <span class="detail-value"><?php echo htmlspecialchars($o['area_name']); ?></span>
                    </div>
                    <div class="detail-row">
                        <span class="detail-label"><?php echo t('quantity'); ?>:</span>
                        <span class="detail-value"><?php echo $o['qty']; ?></span>
                    </div>
                    <div class="detail-row">
                        <span class="detail-label"><?php echo t('status'); ?>:</span>
                        <span class="detail-value">
                            <?php 
                            switch($o['status']){
                                case 'pending': echo '<span class="badge pending">'.t('pending').'</span>'; break;
                                case 'confirmed': echo '<span class="badge confirmed">'.t('confirmed').'</span>'; break;
                                case 'accepted': echo '<span class="badge confirmed">'.t('accepted').'</span>'; break;
                                case 'delivered': echo '<span class="badge delivered">'.t('delivered').'</span>'; break;
                                case 'cancelled': echo '<span class="badge cancelled">'.t('cancelled').'</span>'; break;
                                default: echo '<span class="badge">'.$o['status'].'</span>';
                            }
                            ?>
                        </span>
                    </div>
                </div>
                
                <?php if (!empty($o['driver_name']) && $o['status'] != 'cancelled'): ?>
                <div class="driver-info">
                    <div class="driver-info-title"><?php echo t('driver_info'); ?></div>
                    <div class="driver-detail">
                        <span class="detail-label"><?php echo t('name'); ?>:</span>
                        <span class="detail-value"><?php echo htmlspecialchars($o['driver_name']); ?></span>
                    </div>
                    <div class="driver-detail">
                        <span class="detail-label"><?php echo t('phone'); ?>:</span>
                        <span class="detail-value"><?php echo htmlspecialchars($o['driver_phone']); ?></span>
                    </div>
                    
                    <?php if ($hasDriverLocation): ?>
                    <div style="text-align: center; margin-top: 10px;">
                        <button class="track-driver-btn" 
                                data-order-id="<?php echo $o['id']; ?>"
                                data-driver-name="<?php echo htmlspecialchars($o['driver_name']); ?>"
                                data-driver-phone="<?php echo htmlspecialchars($o['driver_phone']); ?>"
                                data-driver-lat="<?php echo $o['driver_lat']; ?>"
                                data-driver-lng="<?php echo $o['driver_lng']; ?>"
                                data-last-update="<?php echo $lastUpdate; ?>">
                            🚗 <?php echo t('track_driver'); ?>
                        </button>
                        <div class="last-update"><?php echo t('last_updated'); ?>: <?php echo $lastUpdate; ?></div>
                    </div>
                    <?php elseif ($o['status'] == 'accepted'): ?>
                    <div style="text-align: center; margin-top: 10px; color: var(--gray); font-style: italic;">
                        <?php echo t('driver_location_unavailable'); ?>
                    </div>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
                
                <div class="order-actions">
                    <?php if($o['status']=='pending'): ?>
                        <a href="?cancel_id=<?php echo $o['id']; ?>" class="cancel-btn"><?php echo t('cancel'); ?></a>
                    <?php else: ?>
                        <span style="color: var(--gray); font-style: italic; font-size: 0.8rem;"><?php echo t('no_actions'); ?></span>
                    <?php endif; ?>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>
  </div>
</main>

<div class="footer">© <?php echo date('Y'); ?> <?php echo t('propane_service'); ?></div>

<!-- Leaflet JS (OpenStreetMap - Free, no API key required) -->
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>

<script>
  document.addEventListener('DOMContentLoaded', function() {
    // Add animation to order cards with staggered delay
    const cards = document.querySelectorAll('.order-card');
    cards.forEach((card, index) => {
      card.style.animationDelay = `${index * 0.1}s`;
    });

    // Add confirmation for cancel action
    const cancelButtons = document.querySelectorAll('.cancel-btn');
    cancelButtons.forEach(button => {
      button.addEventListener('click', function(e) {
        if (!confirm('Are you sure you want to cancel this order?')) {
          e.preventDefault();
        }
      });
    });

    // Driver location tracking functionality
    let driverMap = null;
    let driverMarker = null;
    let driverLocationInterval = null;
    let currentOrderId = null;

    // Map modal event listeners
    document.getElementById('closeDriverModal').addEventListener('click', closeDriverMapModal);
    document.getElementById('closeDriverMap').addEventListener('click', closeDriverMapModal);
    document.getElementById('refreshDriverLocation').addEventListener('click', refreshDriverLocation);

    // Close modal when clicking outside
    document.getElementById('driverMapModal').addEventListener('click', function(e) {
      if (e.target === this) {
        closeDriverMapModal();
      }
    });

    // Attach event listeners to track driver buttons
    document.querySelectorAll('.track-driver-btn').forEach(button => {
      button.addEventListener('click', function() {
        const orderId = this.getAttribute('data-order-id');
        const driverName = this.getAttribute('data-driver-name');
        const driverPhone = this.getAttribute('data-driver-phone');
        const driverLat = parseFloat(this.getAttribute('data-driver-lat'));
        const driverLng = parseFloat(this.getAttribute('data-driver-lng'));
        const lastUpdate = this.getAttribute('data-last-update');
        
        showDriverMapModal(orderId, driverName, driverPhone, driverLat, driverLng, lastUpdate);
      });
    });

    function showDriverMapModal(orderId, driverName, driverPhone, driverLat, driverLng, lastUpdate) {
      currentOrderId = orderId;
      
      // Update modal content
      document.getElementById('driverModalTitle').textContent = driverName + ' - <?php echo t('location'); ?>';
      document.getElementById('driverName').textContent = driverName;
      document.getElementById('driverPhone').textContent = driverPhone;
      document.getElementById('driverLastUpdate').textContent = '<?php echo t('last_updated'); ?>: ' + lastUpdate;
      document.getElementById('driverDistanceInfo').style.display = 'none';
      
      // Show fallback message while loading
      const fallbackMap = document.getElementById('driverFallbackMap');
      if (fallbackMap) {
        fallbackMap.style.display = 'flex';
        fallbackMap.innerHTML = `
            <div>🌍 Loading driver location...</div>
            <div id="driverLocationStatus" class="location-status warning">
                Loading map...
            </div>
        `;
      }
      
      // Clear any existing map
      if (driverMap) {
        driverMap.remove();
        driverMap = null;
      }
      driverMarker = null;
      
      // Initialize map with slight delay to ensure DOM is ready
      setTimeout(() => {
        try {
          initDriverMap(driverLat, driverLng, driverName);
          startTrackingDriverLocation(orderId);
        } catch (error) {
          console.error('Failed to initialize driver map:', error);
        }
      }, 100);
      
      // Show modal
      document.getElementById('driverMapModal').style.display = 'block';
    }

    function initDriverMap(lat, lng, driverName) {
      try {
        // Remove fallback message
        const fallbackMap = document.getElementById('driverFallbackMap');
        if (fallbackMap) {
          fallbackMap.style.display = 'none';
        }
        
        // Check if map container exists and is proper size
        const mapContainer = document.getElementById('driver-map-container');
        if (!mapContainer) {
          throw new Error('Driver map container not found');
        }
        
        // Initialize Leaflet map
        driverMap = L.map('driver-map-container').setView([lat, lng], 15);
        
        // Add OpenStreetMap tiles
        L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
          attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
          maxZoom: 19
        }).addTo(driverMap);
        
        // Add driver marker
        const driverIcon = L.divIcon({
          className: 'driver-marker',
          html: '<div style="background:#4cc9f0; width:20px; height:20px; border-radius:50%; border:3px solid white; box-shadow:0 2px 8px rgba(76, 201, 240, 0.5);"></div>',
          iconSize: [26, 26],
          iconAnchor: [13, 13]
        });
        
        driverMarker = L.marker([lat, lng], {icon: driverIcon}).addTo(driverMap)
          .bindPopup(`
              <div style="padding: 10px; min-width: 200px;">
                  <h3 style="margin: 0 0 8px 0; color: #333;">🚗 ${driverName}</h3>
                  <p style="margin: 0; font-size: 12px; color: #999;">Driver Location</p>
              </div>
          `)
          .openPopup();
          
        console.log('Driver map initialized successfully');
        
        // Update status
        document.getElementById('driverLocationStatus').className = 'location-status success';
        document.getElementById('driverLocationStatus').innerHTML = '✅ Driver location loaded';
        
      } catch (error) {
        console.error('Driver map initialization error:', error);
        const fallbackMap = document.getElementById('driverFallbackMap');
        if (fallbackMap) {
          fallbackMap.innerHTML = `
              <div style="text-align: center;">
                  <p>🌍 Map Loading Failed</p>
                  <p><small>Coordinates: ${lat}, ${lng}</small></p>
                  <p><small>${error.message}</small></p>
              </div>
          `;
        }
        throw error;
      }
    }

    function startTrackingDriverLocation(orderId) {
      // Clear any existing interval
      if (driverLocationInterval) {
        clearInterval(driverLocationInterval);
      }
      
      // Update driver location every 30 seconds
      driverLocationInterval = setInterval(() => {
        fetch(`get_driver_location.php?order_id=${orderId}`)
          .then(response => response.json())
          .then(data => {
            if (data.success && data.driver_lat && data.driver_lng) {
              // Update marker position
              const newLatLng = [parseFloat(data.driver_lat), parseFloat(data.driver_lng)];
              if (driverMarker) {
                driverMarker.setLatLng(newLatLng);
                
                // Update map view to follow driver
                driverMap.setView(newLatLng, driverMap.getZoom());
              }
              
              // Update last update time
              document.getElementById('driverLastUpdate').textContent = 
                '<?php echo t('last_updated'); ?>: ' + data.last_update;
            }
          })
          .catch(error => {
            console.error('Error fetching driver location:', error);
          });
      }, 30000); // Update every 30 seconds
    }

    function refreshDriverLocation() {
      if (!currentOrderId) return;
      
      document.getElementById('driverLocationStatus').className = 'location-status warning';
      document.getElementById('driverLocationStatus').innerHTML = '🔄 Refreshing driver location...';
      
      fetch(`get_driver_location.php?order_id=${currentOrderId}`)
        .then(response => response.json())
        .then(data => {
          if (data.success && data.driver_lat && data.driver_lng) {
            // Update marker position
            const newLatLng = [parseFloat(data.driver_lat), parseFloat(data.driver_lng)];
            if (driverMarker) {
              driverMarker.setLatLng(newLatLng);
              
              // Update map view to follow driver
              driverMap.setView(newLatLng, driverMap.getZoom());
            }
            
            // Update last update time
            document.getElementById('driverLastUpdate').textContent = 
              '<?php echo t('last_updated'); ?>: ' + data.last_update;
            
            document.getElementById('driverLocationStatus').className = 'location-status success';
            document.getElementById('driverLocationStatus').innerHTML = '✅ Driver location updated';
          } else {
            document.getElementById('driverLocationStatus').className = 'location-status error';
            document.getElementById('driverLocationStatus').innerHTML = '❌ Unable to get driver location';
          }
        })
        .catch(error => {
          document.getElementById('driverLocationStatus').className = 'location-status error';
          document.getElementById('driverLocationStatus').innerHTML = '❌ Error refreshing location';
        });
    }

    function closeDriverMapModal() {
      document.getElementById('driverMapModal').style.display = 'none';
      
      // Clear tracking interval
      if (driverLocationInterval) {
        clearInterval(driverLocationInterval);
        driverLocationInterval = null;
      }
      
      currentOrderId = null;
      
      // Remove map
      if (driverMap) {
        try {
          driverMap.remove();
        } catch (e) {
          console.log('Error removing driver map:', e);
        }
        driverMap = null;
      }
      driverMarker = null;
    }
  });
</script>
</body>
</html>