<?php
session_start();
require 'db.php';
if (!isset($_SESSION['driver_id'])) { header("Location: driver_login.php"); exit; }
$driver_id = $_SESSION['driver_id'];

// Handle Take orders
if (isset($_GET['take'])) {
    $stmt = $pdo->prepare("UPDATE orders SET driver_id=?, status='accepted' WHERE id=? AND status='pending'");
    $stmt->execute([$driver_id, (int)$_GET['take']]);
    header("Location: driver_panel.php");
    exit;
}

// Handle Delivered orders
if (isset($_GET['delivered'])) {
    $orderId = (int)$_GET['delivered'];

    // Check if order is accepted
    $stmtCheck = $pdo->prepare("SELECT status FROM orders WHERE id=? AND driver_id=?");
    $stmtCheck->execute([$orderId, $driver_id]);
    $orderStatus = $stmtCheck->fetchColumn();

    if ($orderStatus === 'accepted') {
        $pdo->beginTransaction();
        try {
            // Mark delivered
            $stmt = $pdo->prepare("UPDATE orders SET status='delivered' WHERE id=? AND driver_id=?");
            $stmt->execute([$orderId, $driver_id]);

            // Increase empty_stock by 1
            $stmt2 = $pdo->prepare("UPDATE drivers 
                                    SET empty_stock = IFNULL(empty_stock,0) + 1
                                    WHERE id=?");
            $stmt2->execute([$driver_id]);

            // Recalculate filled_stock = stock - sold - empty_stock
            $stmt3 = $pdo->prepare("UPDATE drivers 
                                    SET filled_stock = GREATEST(IFNULL(stock,0) - IFNULL(sold,0) - IFNULL(empty_stock,0),0)
                                    WHERE id=?");
            $stmt3->execute([$driver_id]);

            $pdo->commit();
            header("Location: driver_panel.php");
            exit;

        } catch (Exception $e) {
            $pdo->rollBack();
            echo "Error: " . $e->getMessage();
        }
    } else {
        header("Location: driver_panel.php");
        exit;
    }
}

// Handle Manual Stock Update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $exchange = (int)($_POST['exchange'] ?? 0);
    $manual_sell = (int)($_POST['manual_sell'] ?? 0);

    if ($manual_sell > 0) {
        // Manual sell: increase sold, decrease stock
        $stmt = $pdo->prepare("UPDATE drivers 
                               SET sold = IFNULL(sold,0) + ?, 
                                   stock = GREATEST(IFNULL(stock,0) - ?,0)
                               WHERE id=?");
        $stmt->execute([$manual_sell, $manual_sell, $driver_id]);
    }

    if ($exchange > 0) {
        // Manual exchange: decrease sold, increase empty_stock
        $stmt = $pdo->prepare("UPDATE drivers 
                               SET sold = GREATEST(IFNULL(sold,0) - ?,0), 
                                   empty_stock = IFNULL(empty_stock,0) + ?
                               WHERE id=?");
        $stmt->execute([$exchange, $exchange, $driver_id]);
    }

    // Recalculate filled_stock
    $stmt3 = $pdo->prepare("UPDATE drivers 
                            SET filled_stock = GREATEST(IFNULL(stock,0) - IFNULL(sold,0) - IFNULL(empty_stock,0),0)
                            WHERE id=?");
    $stmt3->execute([$driver_id]);

    header("Location: driver_panel.php");
    exit;
}

// Fetch driver info
$stmt = $pdo->prepare("SELECT * FROM drivers WHERE id=?");
$stmt->execute([$driver_id]);
$driver = $stmt->fetch(PDO::FETCH_ASSOC);

// Fetch orders in driver area
$ordersStmt = $pdo->prepare("SELECT o.*, u.name AS user_name, u.address, u.gps_lat, u.gps_lng
                             FROM orders o 
                             JOIN users u ON o.user_id=u.id 
                             WHERE o.area_id=? ORDER BY o.created_at DESC");
$ordersStmt->execute([$driver['area_id']]);
$orders = $ordersStmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html>
<head>
  <meta charset="utf-8">
  <title>Driver Panel</title>
  <link rel="stylesheet" href="assets/css/styles.css">
  <meta http-equiv="refresh" content="8">
</head>
<body>
  <div class="header">
    <div class="header-inner container">
      <div class="brand"><div class="logo"></div><h1>Driver — <?php echo htmlspecialchars($driver['name']); ?></h1></div>
      <a class="btn secondary" href="driver_login.php?logout=1">Logout</a>
    </div>
  </div>

  <main class="container">
    <div class="kpis">
      <div class="kpi"><h3>Main Stock</h3><div><?php echo (int)$driver['stock']; ?></div></div>
      <div class="kpi"><h3>Filled</h3><div><?php echo (int)$driver['filled_stock']; ?></div></div>
      <div class="kpi"><h3>Empty</h3><div><?php echo (int)$driver['empty_stock']; ?></div></div>
      <div class="kpi"><h3>Sold</h3><div><?php echo (int)$driver['sold']; ?></div></div>
      <div class="kpi"><h3>Orders</h3><div><?php echo count($orders); ?></div></div>
      <div class="kpi"><h3>Refresh</h3><div>8s</div></div>
    </div>

    <div class="card" style="margin-top:16px;">
      <h1>Orders in your area</h1>
      <div class="hr"></div>
      <table class="table">
        <thead>
          <tr><th>ID</th><th>User</th><th>Address</th><th>Status</th><th>GPS</th><th>Action</th></tr>
        </thead>
        <tbody>
        <?php foreach($orders as $o): ?>
          <tr>
            <td>#<?php echo $o['id']; ?></td>
            <td><?php echo htmlspecialchars($o['user_name']); ?></td>
            <td><?php echo htmlspecialchars($o['address'] ?? '-'); ?></td>
            <td><span class="badge <?php echo $o['status']; ?>"><?php echo $o['status']; ?></span></td>
            <td>
              <?php if($o['gps_lat'] && $o['gps_lng']): ?>
                <a class="btn secondary" target="_blank"
                   href="https://www.google.com/maps?q=<?php echo $o['gps_lat']; ?>,<?php echo $o['gps_lng']; ?>">Map</a>
              <?php else: ?><span class="muted">—</span><?php endif; ?>
            </td>
            <td>
              <?php if ($o['status']=='pending'): ?>
                <a class="btn" href="?take=<?php echo $o['id']; ?>">Take</a>
              <?php elseif ($o['status']=='accepted' && $o['driver_id']==$driver_id): ?>
                <a class="btn warn" href="?delivered=<?php echo $o['id']; ?>">Mark Delivered</a>
              <?php else: ?>
                <span class="muted">—</span>
              <?php endif; ?>
            </td>
          </tr>
        <?php endforeach; ?>
        </tbody>
      </table>
    </div>

    <div class="card" style="margin-top:16px;">
      <h2>Manual Stock Management</h2>
      <div class="hr"></div>
      <form method="post">
        <div class="grid grid-2">
          <div>
            <label>Exchange Empty Cylinders (Return)</label>
            <input type="number" class="input" name="exchange" min="0" max="<?php echo $driver['sold']; ?>" placeholder="Sold cylinders returned">
          </div>
          <div>
            <label>Manual Sell</label>
            <input type="number" class="input" name="manual_sell" min="0" max="<?php echo $driver['filled_stock']; ?>" placeholder="Sell filled cylinders">
          </div>
          <div style="grid-column:1 / -1">
            <button type="submit" class="btn">Update Stock</button>
          </div>
        </div>
      </form>
    </div>

  </main>

  <div class="footer">© <?php echo date('Y'); ?> Propane Service</div>
</body>
</html>
