<?php
session_start();
require 'db.php';
require 'lang.php';

if (!isset($_SESSION['driver_id'])) { 
    header("Location: driver_login.php"); 
    exit; 
}

$driver_id = $_SESSION['driver_id'];

// Handle AJAX requests
if (isset($_GET['ajax'])) {
    header('Content-Type: application/json');
    
    // Fetch driver info
    $stmt = $pdo->prepare("SELECT * FROM drivers WHERE id=?");
    $stmt->execute([$driver_id]);
    $driver = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Pagination settings
    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $per_page = 20;
    $offset = ($page - 1) * $per_page;

    // Get filter parameters
    $status_filter = $_GET['status'] ?? ['pending', 'accepted'];
    if (!is_array($status_filter)) {
        $status_filter = [$status_filter];
    }

    // Build WHERE clause
    $status_placeholders = implode(',', array_fill(0, count($status_filter), '?'));
    $where_clause = "WHERE o.area_id=?";
    if (!empty($status_filter)) {
        $where_clause .= " AND o.status IN ($status_placeholders)";
    }

    // Add condition for accepted orders
    if (in_array('accepted', $status_filter)) {
        $where_clause .= " AND (o.status != 'accepted' OR o.driver_id = ?)";
        $driver_id_param = $driver_id;
    } else {
        $driver_id_param = null;
    }

    // Get total count for pagination
    $countStmt = $pdo->prepare("SELECT COUNT(*) FROM orders o $where_clause");
    $countParams = [$driver['area_id']];
    if (!empty($status_filter)) {
        $countParams = array_merge($countParams, $status_filter);
    }
    if ($driver_id_param !== null) {
        $countParams[] = $driver_id_param;
    }
    $countStmt->execute($countParams);
    $total_orders = $countStmt->fetchColumn();
    $total_pages = ceil($total_orders / $per_page);

    // Fetch orders
    $ordersStmt = $pdo->prepare("SELECT o.*, u.name AS user_name, u.address, u.gps_lat, u.gps_lng,
                                 d.name AS driver_name
                                 FROM orders o 
                                 JOIN users u ON o.user_id=u.id 
                                 LEFT JOIN drivers d ON o.driver_id = d.id
                                 $where_clause ORDER BY o.created_at DESC
                                 LIMIT $per_page OFFSET $offset");
                                 
    $params = [$driver['area_id']];
    if (!empty($status_filter)) {
        $params = array_merge($params, $status_filter);
    }
    if ($driver_id_param !== null) {
        $params[] = $driver_id_param;
    }
    $ordersStmt->execute($params);
    $orders = $ordersStmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'driver' => $driver,
        'orders' => $orders,
        'total_orders' => $total_orders,
        'total_pages' => $total_pages,
        'current_page' => $page
    ]);
    exit;
}

// Handle Take orders
if (isset($_GET['take'])) {
    $stmt = $pdo->prepare("UPDATE orders SET driver_id=?, status='accepted' WHERE id=? AND status='pending'");
    $stmt->execute([$driver_id, (int)$_GET['take']]);
    header("Location: driver_panel.php");
    exit;
}

// Handle Delivered orders
if (isset($_GET['delivered'])) {
    $orderId = (int)$_GET['delivered'];

    // Check if order is accepted
    $stmtCheck = $pdo->prepare("SELECT status FROM orders WHERE id=? AND driver_id=?");
    $stmtCheck->execute([$orderId, $driver_id]);
    $orderStatus = $stmtCheck->fetchColumn();

    if ($orderStatus === 'accepted') {
        $pdo->beginTransaction();
        try {
            // Mark delivered
            $stmt = $pdo->prepare("UPDATE orders SET status='delivered' WHERE id=? AND driver_id=?");
            $stmt->execute([$orderId, $driver_id]);

            // Increase empty_stock by 1
            $stmt2 = $pdo->prepare("UPDATE drivers 
                                    SET empty_stock = IFNULL(empty_stock,0) + 1
                                    WHERE id=?");
            $stmt2->execute([$driver_id]);

            // Recalculate filled_stock = stock - sold - empty_stock
            $stmt3 = $pdo->prepare("UPDATE drivers 
                                    SET filled_stock = GREATEST(IFNULL(stock,0) - IFNULL(sold,0) - IFNULL(empty_stock,0),0)
                                    WHERE id=?");
            $stmt3->execute([$driver_id]);

            $pdo->commit();
            header("Location: driver_panel.php");
            exit;

        } catch (Exception $e) {
            $pdo->rollBack();
            echo "Error: " . $e->getMessage();
        }
    } else {
        header("Location: driver_panel.php");
        exit;
    }
}

// Handle Manual Stock Update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $exchange = (int)($_POST['exchange'] ?? 0);
    $manual_sell = (int)($_POST['manual_sell'] ?? 0);

    if ($manual_sell > 0) {
        // Manual sell: increase sold, decrease stock
        $stmt = $pdo->prepare("UPDATE drivers 
                               SET sold = IFNULL(sold,0) + ?, 
                                   stock = GREATEST(IFNULL(stock,0) - ?,0)
                               WHERE id=?");
        $stmt->execute([$manual_sell, $manual_sell, $driver_id]);
    }

    if ($exchange > 0) {
        // Manual exchange: decrease sold, increase empty_stock
        $stmt = $pdo->prepare("UPDATE drivers 
                               SET sold = GREATEST(IFNULL(sold,0) - ?,0), 
                                   empty_stock = IFNULL(empty_stock,0) + ?
                               WHERE id=?");
        $stmt->execute([$exchange, $exchange, $driver_id]);
    }

    // Recalculate filled_stock
    $stmt3 = $pdo->prepare("UPDATE drivers 
                            SET filled_stock = GREATEST(IFNULL(stock,0) - IFNULL(sold,0) - IFNULL(empty_stock,0),0)
                            WHERE id=?");
    $stmt3->execute([$driver_id]);

    header("Location: driver_panel.php");
    exit;
}

// Fetch driver info for initial page load
$stmt = $pdo->prepare("SELECT * FROM drivers WHERE id=?");
$stmt->execute([$driver_id]);
$driver = $stmt->fetch(PDO::FETCH_ASSOC);

// Set direction based on language
$dir = ($lang == 'ar') ? 'rtl' : 'ltr';
?>
<!DOCTYPE html>
<html dir="<?php echo $dir; ?>">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title><?php echo t('driver_panel'); ?> — Propane Service</title>
  <style>
    :root {
      --primary: #4361ee;
      --secondary: #3a0ca3;
      --success: #4cc9f0;
      --danger: #f72585;
      --warning: #fca311;
      --light: #f8f9fa;
      --dark: #212529;
      --gray: #6c757d;
      --light-gray: #e9ecef;
      --border-radius: 12px;
      --box-shadow: 0 10px 25px rgba(0,0,0,0.1);
      --transition: all 0.3s ease;
    }

    * {
      margin: 0;
      padding: 0;
      box-sizing: border-box;
      font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    }

    body {
      background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
      min-height: 100vh;
      padding: 20px;
      color: var(--dark);
      direction: <?php echo $dir; ?>;
      overflow-x: hidden;
    }

    .header {
      background: white;
      border-radius: var(--border-radius);
      box-shadow: var(--box-shadow);
      margin-bottom: 30px;
      padding: 15px 25px;
      animation: slideDown 0.5s ease;
      display: flex;
      justify-content: space-between;
      align-items: center;
      flex-wrap: wrap;
      gap: 15px;
    }

    .brand {
      display: flex;
      align-items: center;
      gap: 15px;
    }

    .logo {
      width: 40px;
      height: 40px;
      background: var(--primary);
      border-radius: 50%;
      display: flex;
      align-items: center;
      justify-content: center;
      color: white;
      font-weight: bold;
    }

    h1, h2, h3 {
      color: var(--primary);
    }

    .language-switcher {
      display: flex;
      gap: 10px;
    }

    .lang-btn {
      padding: 5px 10px;
      border: 1px solid var(--primary);
      border-radius: var(--border-radius);
      background: white;
      color: var(--primary);
      cursor: pointer;
      transition: var(--transition);
      text-decoration: none;
      font-size: 0.9rem;
    }

    .lang-btn.active {
      background: var(--primary);
      color: white;
    }

    .lang-btn:hover {
      background: var(--primary);
      color: white;
    }

    .btn {
      display: inline-block;
      padding: 10px 20px;
      background: var(--primary);
      color: white;
      border: none;
      border-radius: var(--border-radius);
      cursor: pointer;
      text-decoration: none;
      font-weight: 600;
      transition: var(--transition);
      box-shadow: 0 4px 6px rgba(67, 97, 238, 0.3);
    }

    .btn:hover {
      transform: translateY(-2px);
      box-shadow: 0 6px 12px rgba(67, 97, 238, 0.4);
    }

    .btn.secondary {
      background: white;
      color: var(--primary);
      border: 1px solid var(--primary);
    }

    .btn.secondary:hover {
      background: var(--primary);
      color: white;
    }

    .btn.warn {
      background: var(--warning);
      color: white;
    }

    .btn.warn:hover {
      background: #e69500;
    }

    .btn.success {
      background: var(--success);
      color: white;
    }

    .btn.success:hover {
      background: #3aa5c9;
    }

    .btn.small {
      padding: 6px 12px;
      font-size: 0.85rem;
    }

    .container {
      max-width: 1400px;
      margin: 0 auto;
      padding: 0 15px;
    }

    .kpis {
      display: grid;
      grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
      gap: 15px;
      margin-bottom: 25px;
    }

    .kpi {
      background: white;
      border-radius: var(--border-radius);
      box-shadow: var(--box-shadow);
      padding: 20px;
      text-align: center;
      transition: var(--transition);
      animation: fadeIn 0.6s ease;
    }

    .kpi:hover {
      transform: translateY(-5px);
      box-shadow: 0 15px 30px rgba(0,0,0,0.15);
    }

    .kpi h3 {
      margin-bottom: 10px;
      font-size: 0.9rem;
      color: var(--gray);
    }

    .kpi div {
      font-size: 1.8rem;
      font-weight: bold;
      color: var(--primary);
    }

    .card {
      background: white;
      border-radius: var(--border-radius);
      box-shadow: var(--box-shadow);
      padding: 25px;
      margin-bottom: 25px;
      transition: var(--transition);
      animation: fadeIn 0.6s ease;
      overflow: hidden;
    }

    .card:hover {
      transform: translateY(-5px);
      box-shadow: 0 15px 30px rgba(0,0,0,0.15);
    }

    .hr {
      height: 1px;
      background: var(--light-gray);
      margin: 20px 0;
    }

    .table-container {
      overflow-x: auto;
      margin-top: 20px;
      max-width: 100%;
    }

    .table {
      width: 100%;
      border-collapse: collapse;
      min-width: 1000px;
    }

    .table th, .table td {
      padding: 12px 15px;
      text-align: left;
      border-bottom: 1px solid var(--light-gray);
      white-space: nowrap;
    }

    .table th {
      background: var(--light);
      font-weight: 600;
      position: sticky;
      top: 0;
    }

    .table tr {
      transition: var(--transition);
    }

    .table tr:hover {
      background: rgba(67, 97, 238, 0.05);
    }

    .badge {
      padding: 6px 12px;
      border-radius: 20px;
      font-size: 0.85rem;
      font-weight: 600;
      display: inline-block;
    }

    .pending {
      background: rgba(252, 163, 17, 0.2);
      color: #b36a00;
    }

    .accepted {
      background: rgba(67, 97, 238, 0.2);
      color: var(--primary);
    }

    .delivered {
      background: rgba(76, 201, 240, 0.2);
      color: #0a6c74;
    }

    .cancelled {
      background: rgba(247, 37, 133, 0.2);
      color: #a01a58;
    }

    .muted {
      color: var(--gray);
    }

    .grid {
      display: grid;
      gap: 20px;
    }

    .grid-2 {
      grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    }

    label {
      display: block;
      margin-bottom: 8px;
      font-weight: 600;
      color: var(--dark);
    }

    .input {
      width: 100%;
      padding: 12px 15px;
      border: 1px solid var(--light-gray);
      border-radius: var(--border-radius);
      font-size: 1rem;
      transition: var(--transition);
    }

    .input:focus {
      outline: none;
      border-color: var(--primary);
      box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
    }

    .footer {
      text-align: center;
      padding: 20px;
      color: var(--gray);
      margin-top: 40px;
    }

    .refresh-indicator {
      display: inline-block;
      padding: 4px 8px;
      background: var(--primary);
      color: white;
      border-radius: 20px;
      font-size: 0.8rem;
      animation: pulse 2s infinite;
    }

    .filter-tabs {
      display: flex;
      flex-wrap: wrap;
      gap: 10px;
      margin-bottom: 20px;
      background: var(--light);
      padding: 15px;
      border-radius: var(--border-radius);
    }

    .filter-tab {
      padding: 8px 16px;
      border-radius: 20px;
      cursor: pointer;
      transition: var(--transition);
      font-weight: 500;
    }

    .filter-tab.active {
      background: var(--primary);
      color: white;
    }

    .filter-tab:hover:not(.active) {
      background: var(--light-gray);
    }

    .pagination {
      display: flex;
      justify-content: center;
      gap: 8px;
      margin-top: 20px;
      flex-wrap: wrap;
    }

    .pagination-btn {
      padding: 8px 14px;
      border-radius: var(--border-radius);
      background: white;
      border: 1px solid var(--light-gray);
      cursor: pointer;
      transition: var(--transition);
    }

    .pagination-btn.active {
      background: var(--primary);
      color: white;
      border-color: var(--primary);
    }

    .pagination-btn:hover:not(.active) {
      background: var(--light-gray);
    }

    .order-info {
      display: grid;
      grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
      gap: 10px;
      margin-bottom: 10px;
    }

    .order-info-item {
      display: flex;
      flex-direction: column;
    }

    .order-info-label {
      font-size: 0.85rem;
      color: var(--gray);
      margin-bottom: 4px;
    }

    .order-info-value {
      font-weight: 500;
    }

    .live-indicator {
      display: inline-flex;
      align-items: center;
      gap: 5px;
      padding: 4px 8px;
      background: var(--success);
      color: white;
      border-radius: 20px;
      font-size: 0.8rem;
    }

    .live-dot {
      width: 8px;
      height: 8px;
      background: white;
      border-radius: 50%;
      animation: pulseLive 1.5s infinite;
    }

    @keyframes slideDown {
      from {
        opacity: 0;
        transform: translateY(-20px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    @keyframes fadeIn {
      from {
        opacity: 0;
        transform: translateY(20px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    @keyframes pulse {
      0% {
        box-shadow: 0 0 0 0 rgba(67, 97, 238, 0.4);
      }
      70% {
        box-shadow: 0 0 0 10px rgba(67, 97, 238, 0);
      }
      100% {
        box-shadow: 0 0 0 0 rgba(67, 97, 238, 0);
      }
    }

    @keyframes pulseLive {
      0% {
        opacity: 1;
      }
      50% {
        opacity: 0.5;
      }
      100% {
        opacity: 1;
      }
    }

    @media (max-width: 768px) {
      .header {
        flex-direction: column;
        align-items: flex-start;
      }
      
      .kpis {
        grid-template-columns: repeat(2, 1fr);
      }
      
      .filter-tabs {
        flex-direction: column;
      }
      
      .pagination {
        gap: 5px;
      }
      
      .pagination-btn {
        padding: 6px 10px;
        font-size: 0.9rem;
      }
    }
  </style>
  <script>
    document.addEventListener('DOMContentLoaded', function() {
      let currentPage = <?php echo isset($_GET['page']) ? (int)$_GET['page'] : 1; ?>;
      let currentStatus = '<?php echo isset($_GET['status']) ? $_GET['status'] : "pending,accepted"; ?>';
      let refreshInterval = 3000; // 3 seconds
      let refreshTimer;

      // Initialize the page
      function initPage() {
        loadData();
        setupEventListeners();
        startAutoRefresh();
      }

      // Load data via AJAX
      function loadData() {
        const url = new URL(window.location.href);
        url.searchParams.set('ajax', '1');
        url.searchParams.set('page', currentPage);
        if (currentStatus) {
          url.searchParams.set('status', currentStatus);
        }

        fetch(url.toString())
          .then(response => response.json())
          .then(data => {
            updateKPIs(data.driver);
            updateOrdersTable(data.orders, data.total_orders, data.total_pages, data.current_page);
          })
          .catch(error => {
            console.error('Error loading data:', error);
          });
      }

      // Update KPIs with new data
      function updateKPIs(driver) {
        document.querySelector('[data-kpi="stock"]').textContent = driver.stock || 0;
        document.querySelector('[data-kpi="filled"]').textContent = driver.filled_stock || 0;
        document.querySelector('[data-kpi="empty"]').textContent = driver.empty_stock || 0;
        document.querySelector('[data-kpi="sold"]').textContent = driver.sold || 0;
      }

      // Update orders table with new data
      function updateOrdersTable(orders, totalOrders, totalPages, currentPage) {
        const tbody = document.querySelector('.table tbody');
        
        if (orders.length === 0) {
          tbody.innerHTML = `
            <tr>
              <td colspan="7" style="text-align: center; padding: 20px;">
                <?php echo t('no_orders_match'); ?>
              </td>
            </tr>
          `;
          return;
        }

        let html = '';
        orders.forEach(order => {
          const isMyOrder = order.driver_id == <?php echo $driver_id; ?>;
          
          html += `
            <tr>
              <td>#${order.id}</td>
              <td>
                <div class="order-info">
                  <div class="order-info-item">
                    <span class="order-info-label"><?php echo t('name'); ?></span>
                    <span class="order-info-value">${escapeHtml(order.user_name)}</span>
                  </div>
                </div>
              </td>
              <td>
                <div class="order-info">
                  <div class="order-info-item">
                    <span class="order-info-label"><?php echo t('address'); ?></span>
                    <span class="order-info-value">${escapeHtml(order.address || '-')}</span>
                  </div>
                </div>
              </td>
              <td>
                <span class="badge ${order.status}">
                  <?php echo t('${order.status}'); ?>
                </span>
              </td>
              <td>
                ${order.driver_name ? `
                  <div class="order-info">
                    <div class="order-info-item">
                      <span class="order-info-label"><?php echo t('driver'); ?></span>
                      <span class="order-info-value">${escapeHtml(order.driver_name)}</span>
                    </div>
                  </div>
                  ${isMyOrder ? `<span class="badge" style="background: var(--primary); color: white; margin-top: 5px;"><?php echo t('you'); ?></span>` : ''}
                ` : `<span class="muted">—</span>`}
              </td>
              <td>
                ${order.gps_lat && order.gps_lng ? `
                  <a class="btn secondary small" target="_blank"
                     href="https://www.google.com/maps?q=${order.gps_lat},${order.gps_lng}">
                    <?php echo t('map'); ?>
                  </a>
                ` : `<span class="muted">—</span>`}
              </td>
              <td>
                ${order.status == 'pending' ? `
                  <a class="btn small" href="?lang=<?php echo $lang; ?>&status=${currentStatus}&take=${order.id}"><?php echo t('take'); ?></a>
                ` : order.status == 'accepted' && isMyOrder ? `
                  <a class="btn success small" href="?lang=<?php echo $lang; ?>&status=${currentStatus}&delivered=${order.id}"><?php echo t('mark_delivered'); ?></a>
                ` : `<span class="muted">—</span>`}
              </td>
            </tr>
          `;
        });

        tbody.innerHTML = html;
        
        // Update pagination
        updatePagination(totalPages, currentPage);
      }

      // Update pagination controls
      function updatePagination(totalPages, currentPage) {
        const paginationContainer = document.querySelector('.pagination');
        if (totalPages <= 1) {
          paginationContainer.innerHTML = '';
          return;
        }

        let html = '';
        
        if (currentPage > 1) {
          html += `<a href="#" data-page="${currentPage - 1}" class="pagination-btn"><?php echo t('previous'); ?></a>`;
        }
        
        for (let i = 1; i <= totalPages; i++) {
          html += `<a href="#" data-page="${i}" class="pagination-btn ${i == currentPage ? 'active' : ''}">${i}</a>`;
        }
        
        if (currentPage < totalPages) {
          html += `<a href="#" data-page="${currentPage + 1}" class="pagination-btn"><?php echo t('next'); ?></a>`;
        }
        
        paginationContainer.innerHTML = html;
        
        // Add event listeners to pagination buttons
        document.querySelectorAll('.pagination-btn').forEach(btn => {
          btn.addEventListener('click', function(e) {
            e.preventDefault();
            currentPage = parseInt(this.getAttribute('data-page'));
            loadData();
          });
        });
      }

      // Set up event listeners
      function setupEventListeners() {
        // Filter tabs
        document.querySelectorAll('.filter-tab').forEach(tab => {
          tab.addEventListener('click', function() {
            const status = this.getAttribute('data-status');
            currentStatus = status;
            currentPage = 1;
            
            // Update active tab
            document.querySelectorAll('.filter-tab').forEach(t => t.classList.remove('active'));
            this.classList.add('active');
            
            loadData();
          });
        });

        // Manual refresh button
        document.getElementById('manual-refresh').addEventListener('click', function() {
          loadData();
        });
      }

      // Start auto-refresh timer
      function startAutoRefresh() {
        clearInterval(refreshTimer);
        refreshTimer = setInterval(loadData, refreshInterval);
      }

      // Stop auto-refresh timer
      function stopAutoRefresh() {
        clearInterval(refreshTimer);
      }

      // Utility function to escape HTML
      function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
      }

      // Initialize the page
      initPage();

      // Clean up on page unload
      window.addEventListener('beforeunload', stopAutoRefresh);
    });
  </script>
</head>
<body>
  <div class="header">
    <div class="brand">
      <div class="logo">D</div>
      <h1><?php echo t('driver_panel'); ?> — <?php echo htmlspecialchars($driver['name']); ?></h1>
    </div>
    
    <div style="display: flex; align-items: center; gap: 15px;">
      <div class="language-switcher">
        <a href="?lang=en" class="lang-btn <?php echo $lang == 'en' ? 'active' : ''; ?>">EN</a>
        <a href="?lang=ar" class="lang-btn <?php echo $lang == 'ar' ? 'active' : ''; ?>">AR</a>
        <a href="?lang=ku" class="lang-btn <?php echo $lang == 'ku' ? 'active' : ''; ?>">KU</a>
      </div>
      <button id="manual-refresh" class="btn secondary"><?php echo t('refresh'); ?></button>
      <a class="btn secondary" href="driver_login.php?logout=1"><?php echo t('logout'); ?></a>
    </div>
  </div>

  <main class="container">
    <div class="kpis">
      <div class="kpi"><h3><?php echo t('main_stock'); ?></h3><div data-kpi="stock"><?php echo (int)$driver['stock']; ?></div></div>
      <div class="kpi"><h3><?php echo t('filled'); ?></h3><div data-kpi="filled"><?php echo (int)$driver['filled_stock']; ?></div></div>
      <div class="kpi"><h3><?php echo t('empty'); ?></h3><div data-kpi="empty"><?php echo (int)$driver['empty_stock']; ?></div></div>
      <div class="kpi"><h3><?php echo t('sold'); ?></h3><div data-kpi="sold"><?php echo (int)$driver['sold']; ?></div></div>
      <div class="kpi"><h3><?php echo t('status'); ?></h3><div class="live-indicator"><span class="live-dot"></span> Live</div></div>
    </div>

    <div class="card">
      <h1><?php echo t('orders_in_area'); ?></h1>
      
      <!-- Status Filter Tabs -->
      <div class="filter-tabs">

        <div class="filter-tab <?php echo isset($_GET['status']) && $_GET['status'] === 'pending' ? 'active' : ''; ?>" data-status="pending">
          <?php echo t('pending'); ?>
        </div>
        <div class="filter-tab <?php echo isset($_GET['status']) && $_GET['status'] === 'accepted' ? 'active' : ''; ?>" data-status="accepted">
          <?php echo t('my_accepted'); ?>
        </div>
        <div class="filter-tab <?php echo isset($_GET['status']) && $_GET['status'] === 'delivered' ? 'active' : ''; ?>" data-status="delivered">
          <?php echo t('delivered'); ?>
        </div>

      </div>
      
      <div class="hr"></div>
      
      <div class="table-container">
        <table class="table">
          <thead>
            <tr>
              <th>ID</th>
              <th><?php echo t('user'); ?></th>
              <th><?php echo t('address'); ?></th>
              <th><?php echo t('status'); ?></th>
              <th><?php echo t('driver'); ?></th>
              <th><?php echo t('gps'); ?></th>
              <th><?php echo t('actions'); ?></th>
            </tr>
          </thead>
          <tbody>
            <!-- Content will be loaded via AJAX -->
          </tbody>
        </table>
      </div>

      <!-- Pagination -->
      <div class="pagination">
        <!-- Content will be loaded via AJAX -->
      </div>
    </div>

    <div class="card">
      <h2><?php echo t('manual_stock_management'); ?></h2>
      <div class="hr"></div>
      <form method="post">
        <input type="hidden" name="lang" value="<?php echo $lang; ?>">
        <div class="grid grid-2">
          <div>
            <label><?php echo t('exchange_empty_cylinders'); ?></label>
            <input type="number" class="input" name="exchange" min="0" max="<?php echo $driver['sold']; ?>" 
                   value="0" placeholder="<?php echo t('sold_cylinders_returned'); ?>">
          </div>
          <div>
            <label><?php echo t('manual_sell'); ?></label>
            <input type="number" class="input" name="manual_sell" min="0" max="<?php echo $driver['filled_stock']; ?>" 
                   value="0" placeholder="<?php echo t('sell_filled_cylinders'); ?>">
          </div>
          <div style="grid-column:1 / -1">
            <button type="submit" class="btn"><?php echo t('update_stock'); ?></button>
          </div>
        </div>
      </form>
    </div>
  </main>

  <div class="footer">© <?php echo date('Y'); ?> <?php echo t('propane_service'); ?></div>
</body>
</html>