<?php
session_start();
require 'db.php';
require 'lang.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: create_profile.php");
    exit;
}

$user_id = $_SESSION['user_id'];

// Fetch current user info
$stmt = $pdo->prepare("SELECT * FROM users WHERE id=?");
$stmt->execute([$user_id]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);

// Handle update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $stmt = $pdo->prepare("UPDATE users 
        SET name=?, phone=?, address=?, area_id=?, gps_lat=?, gps_lng=? 
        WHERE id=?");
    $stmt->execute([
        $_POST['name'], 
        $_POST['phone'], 
        $_POST['address'], 
        $_POST['area_id'],
        $_POST['gps_lat'], 
        $_POST['gps_lng'], 
        $user_id
    ]);
    header("Location: profile.php?updated=1");
    exit;
}

// Load areas
$areas = $pdo->query("SELECT * FROM areas")->fetchAll(PDO::FETCH_ASSOC);

// Set direction based on language
$dir = ($lang == 'ar') ? 'rtl' : 'ltr';
?>
<!DOCTYPE html>
<html dir="<?php echo $dir; ?>">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title><?php echo t('edit_profile'); ?> - Propane Service</title>
  <style>
    :root {
      --primary: #4361ee;
      --secondary: #3a0ca3;
      --success: #4cc9f0;
      --danger: #f72585;
      --warning: #fca311;
      --light: #f8f9fa;
      --dark: #212529;
      --gray: #6c757d;
      --light-gray: #e9ecef;
      --border-radius: 12px;
      --box-shadow: 0 10px 25px rgba(0,0,0,0.1);
      --transition: all 0.3s ease;
    }

    * {
      margin: 0;
      padding: 0;
      box-sizing: border-box;
      font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    }

    body {
      background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
      min-height: 100vh;
      padding: 20px;
      color: var(--dark);
      direction: <?php echo $dir; ?>;
    }

    .container {
      max-width: 1200px;
      margin: 0 auto;
      padding: 0 15px;
    }

    .header {
      background: white;
      border-radius: var(--border-radius);
      box-shadow: var(--box-shadow);
      margin-bottom: 30px;
      padding: 15px 25px;
      animation: slideDown 0.5s ease;
      display: flex;
      justify-content: space-between;
      align-items: center;
      flex-wrap: wrap;
      gap: 15px;
    }

    .brand {
      display: flex;
      align-items: center;
      gap: 15px;
    }

    .logo {
      width: 40px;
      height: 40px;
      background: var(--primary);
      border-radius: 50%;
      display: flex;
      align-items: center;
      justify-content: center;
      color: white;
      font-weight: bold;
    }

    h1 {
      font-size: 1.8rem;
      color: var(--primary);
    }

    .language-switcher {
      display: flex;
      gap: 10px;
      margin-left: auto;
      margin-right: 20px;
    }

    .lang-btn {
      padding: 5px 10px;
      border: 1px solid var(--primary);
      border-radius: var(--border-radius);
      background: white;
      color: var(--primary);
      cursor: pointer;
      transition: var(--transition);
    }

    .lang-btn.active {
      background: var(--primary);
      color: white;
    }

    .lang-btn:hover {
      background: var(--primary);
      color: white;
    }

    .btn {
      display: inline-block;
      padding: 10px 20px;
      background: var(--primary);
      color: white;
      border: none;
      border-radius: var(--border-radius);
      cursor: pointer;
      text-decoration: none;
      font-weight: 600;
      transition: var(--transition);
      box-shadow: 0 4px 6px rgba(67, 97, 238, 0.3);
    }

    .btn:hover {
      transform: translateY(-2px);
      box-shadow: 0 6px 12px rgba(67, 97, 238, 0.4);
    }

    .btn.secondary {
      background: white;
      color: var(--primary);
      border: 1px solid var(--primary);
    }

    .btn.secondary:hover {
      background: var(--primary);
      color: white;
    }

    .btn.block {
      display: block;
      width: 100%;
      text-align: center;
    }

    .card {
      background: white;
      border-radius: var(--border-radius);
      box-shadow: var(--box-shadow);
      padding: 25px;
      margin-bottom: 25px;
      animation: fadeIn 0.6s ease;
    }

    .card h1 {
      margin-bottom: 15px;
      font-size: 1.8rem;
    }

    .hr {
      height: 1px;
      background: var(--light-gray);
      margin: 20px 0;
    }

    .grid {
      display: grid;
      gap: 20px;
    }

    .grid-2 {
      grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    }

    label {
      display: block;
      margin-bottom: 8px;
      font-weight: 600;
      color: var(--dark);
    }

    .input, select {
      width: 100%;
      padding: 12px 15px;
      border: 1px solid var(--light-gray);
      border-radius: var(--border-radius);
      font-size: 1rem;
      transition: var(--transition);
    }

    .input:focus, select:focus {
      outline: none;
      border-color: var(--primary);
      box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
    }

    .success {
      background: rgba(76, 201, 240, 0.2);
      color: #0a6c74;
      padding: 12px;
      border-radius: var(--border-radius);
      margin-bottom: 20px;
      border-left: 4px solid var(--success);
    }

    .muted {
      color: var(--gray);
      font-size: 0.9rem;
    }

    .footer {
      text-align: center;
      padding: 20px;
      color: var(--gray);
      margin-top: 40px;
    }

    @keyframes slideDown {
      from {
        opacity: 0;
        transform: translateY(-20px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    @keyframes fadeIn {
      from {
        opacity: 0;
      }
      to {
        opacity: 1;
      }
    }

    @media (max-width: 768px) {
      .header {
        flex-direction: column;
        align-items: flex-start;
      }
      
      .language-switcher {
        margin: 10px 0;
      }
      
      .grid-2 {
        grid-template-columns: 1fr;
      }
    }
  </style>
  <script>
    document.addEventListener('DOMContentLoaded', function() {
      // Add animation to form elements
      const inputs = document.querySelectorAll('.input, select');
      inputs.forEach((input, index) => {
        input.style.animation = `fadeIn 0.5s ease ${index * 0.1}s forwards`;
        input.style.opacity = 0;
      });

      // Success message animation
      const successMsg = document.querySelector('.success');
      if (successMsg) {
        successMsg.style.animation = 'slideDown 0.5s ease';
      }
    });

    function getGPSInto(latField, lngField, statusField) {
        const statusEl = document.querySelector(statusField);
        if (!navigator.geolocation) {
            statusEl.textContent = "Geolocation not supported";
            return;
        }
        statusEl.textContent = "Fetching location...";
        navigator.geolocation.getCurrentPosition(
            (pos) => {
                document.querySelector(latField).value = pos.coords.latitude;
                document.querySelector(lngField).value = pos.coords.longitude;
                statusEl.textContent = "Location set!";
                
                // Add success animation
                statusEl.style.color = 'green';
                setTimeout(() => {
                  statusEl.style.opacity = 0;
                  setTimeout(() => statusEl.textContent = "", 600);
                }, 2000);
            },
            (err) => {
                statusEl.textContent = "Unable to fetch location";
                statusEl.style.color = 'red';
            }
        );
    }
  </script>
</head>
<body>
  <div class="header">
    <div class="brand">
      <div class="logo">PS</div>
      <h1><?php echo t('propane_service'); ?></h1>
    </div>
    

    
    <a class="btn secondary" href="user_dashboard.php"><?php echo t('dashboard_link'); ?></a>
  </div>

  <main class="container">
    <div class="card">
      <h1><?php echo t('edit_profile'); ?></h1>
      <?php if(isset($_GET['updated'])): ?>
        <p class="success">✅ <?php echo t('profile_updated'); ?></p>
      <?php endif; ?>
      <div class="hr"></div>
      <form method="post" class="grid grid-2">
        <div>
          <label><?php echo t('name'); ?></label>
          <input class="input" type="text" name="name" 
                 value="<?php echo htmlspecialchars($user['name']); ?>" required>
        </div>
        <div>
          <label><?php echo t('phone'); ?></label>
          <input class="input" type="text" name="phone" 
                 value="<?php echo htmlspecialchars($user['phone']); ?>">
        </div>
        <div class="grid-2" style="grid-column:1 / -1">
          <div>
            <label><?php echo t('address'); ?></label>
            <input class="input" type="text" name="address" 
                   value="<?php echo htmlspecialchars($user['address']); ?>">
          </div>
          <div>
            <label><?php echo t('area'); ?></label>
            <select class="input" name="area_id">
              <?php foreach($areas as $a): ?>
                <option value="<?php echo $a['id']; ?>" 
                  <?php echo ($a['id']==$user['area_id'])?'selected':''; ?>>
                  <?php echo htmlspecialchars($a['name']); ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
        </div>

        <div>
          <label><?php echo t('gps_latitude'); ?></label>
          <input id="gps_lat" class="input" type="text" 
                 name="gps_lat" value="<?php echo htmlspecialchars($user['gps_lat']); ?>">
        </div>
        <div>
          <label><?php echo t('gps_longitude'); ?></label>
          <input id="gps_lng" class="input" type="text" 
                 name="gps_lng" value="<?php echo htmlspecialchars($user['gps_lng']); ?>">
        </div>

        <div style="grid-column:1 / -1;display:flex;gap:10px;align-items:center">
          <button class="btn" type="button" 
                  onclick="getGPSInto('#gps_lat','#gps_lng','#gps_status')">
            <?php echo t('use_my_location'); ?>
          </button>
          <span id="gps_status" class="muted"></span>
        </div>

        <div style="grid-column:1 / -1">
          <button class="btn block" type="submit"><?php echo t('update_profile'); ?></button>
        </div>
      </form>
    </div>
  </main>
    <div class="language-switcher">
      <a href="?lang=en" class="lang-btn <?php echo $lang == 'en' ? 'active' : ''; ?>">EN</a>
      <a href="?lang=ar" class="lang-btn <?php echo $lang == 'ar' ? 'active' : ''; ?>">AR</a>
      <a href="?lang=ku" class="lang-btn <?php echo $lang == 'ku' ? 'active' : ''; ?>">KU</a>
    </div>
  <div class="footer">© <?php echo date('Y'); ?> <?php echo t('propane_service'); ?></div>
</body>
</html>