<?php
session_start();
require 'db.php';
require 'lang.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

$user_id = $_SESSION['user_id'];
$message = '';

// Cancel only pending orders
if (isset($_GET['cancel_id'])) {
    $cancel_id = (int)$_GET['cancel_id'];
    $stmt = $pdo->prepare("UPDATE orders SET status='cancelled' WHERE id=? AND user_id=? AND status='pending'");
    $stmt->execute([$cancel_id, $user_id]);
    if ($stmt->rowCount()) {
        $message = "✅ Order #$cancel_id has been cancelled!";
    } else {
        $message = "⚠️ Only pending orders can be cancelled.";
    }
}

// Fetch all non-cancelled orders
$stmt = $pdo->prepare("
    SELECT o.*, a.name AS area_name
    FROM orders o
    LEFT JOIN areas a ON o.area_id = a.id
    WHERE o.user_id = ? AND o.status != 'cancelled'
    ORDER BY o.created_at DESC
");
$stmt->execute([$user_id]);
$orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Set direction based on language
$dir = ($lang == 'ar') ? 'rtl' : 'ltr';
?>
<!DOCTYPE html>
<html dir="<?php echo $dir; ?>">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title><?php echo t('orders'); ?> - Propane Service</title>
<style>
    :root {
      --primary: #4361ee;
      --secondary: #3a0ca3;
      --success: #4cc9f0;
      --danger: #f72585;
      --warning: #fca311;
      --light: #f8f9fa;
      --dark: #212529;
      --gray: #6c757d;
      --light-gray: #e9ecef;
      --border-radius: 12px;
      --box-shadow: 0 10px 25px rgba(0,0,0,0.1);
      --transition: all 0.3s ease;
    }

    * {
      margin: 0;
      padding: 0;
      box-sizing: border-box;
      font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    }

    body {
      background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
      min-height: 100vh;
      padding: 20px;
      color: var(--dark);
      direction: <?php echo $dir; ?>;
    }

    .header {
      background: white;
      border-radius: var(--border-radius);
      box-shadow: var(--box-shadow);
      margin-bottom: 30px;
      padding: 15px 25px;
      animation: slideDown 0.5s ease;
      display: flex;
      justify-content: space-between;
      align-items: center;
      flex-wrap: wrap;
      gap: 15px;
    }

    .brand {
      display: flex;
      align-items: center;
      gap: 15px;
    }

    .logo {
      width: 40px;
      height: 40px;
      background: var(--primary);
      border-radius: 50%;
      display: flex;
      align-items: center;
      justify-content: center;
      color: white;
      font-weight: bold;
    }

    h1, h2 {
      color: var(--primary);
    }

    .language-switcher {
      display: flex;
      gap: 10px;
    }

    .lang-btn {
      padding: 5px 10px;
      border: 1px solid var(--primary);
      border-radius: var(--border-radius);
      background: white;
      color: var(--primary);
      cursor: pointer;
      transition: var(--transition);
      text-decoration: none;
      font-size: 0.9rem;
    }

    .lang-btn.active {
      background: var(--primary);
      color: white;
    }

    .lang-btn:hover {
      background: var(--primary);
      color: white;
    }

    .btn {
      display: inline-block;
      padding: 10px 20px;
      background: var(--primary);
      color: white;
      border: none;
      border-radius: var(--border-radius);
      cursor: pointer;
      text-decoration: none;
      font-weight: 600;
      transition: var(--transition);
      box-shadow: 0 4px 6px rgba(67, 97, 238, 0.3);
    }

    .btn:hover {
      transform: translateY(-2px);
      box-shadow: 0 6px 12px rgba(67, 97, 238, 0.4);
    }

    .btn.secondary {
      background: white;
      color: var(--primary);
      border: 1px solid var(--primary);
    }

    .btn.secondary:hover {
      background: var(--primary);
      color: white;
    }

    .container {
      max-width: 1200px;
      margin: 0 auto;
      padding: 0 15px;
    }

    .card {
      background: white;
      border-radius: var(--border-radius);
      box-shadow: var(--box-shadow);
      padding: 25px;
      margin-bottom: 25px;
      animation: fadeIn 0.6s ease;
    }

    .hr {
      height: 1px;
      background: var(--light-gray);
      margin: 20px 0;
    }

    .new-order-msg {
      background: rgba(76, 201, 240, 0.2);
      padding: 15px;
      margin-bottom: 20px;
      border-radius: var(--border-radius);
      border-left: 4px solid var(--success);
      animation: slideDown 0.5s ease;
    }

    .table {
      width: 100%;
      border-collapse: collapse;
      margin-top: 20px;
    }

    .table th, .table td {
      padding: 12px 15px;
      text-align: left;
      border-bottom: 1px solid var(--light-gray);
    }

    .table th {
      background: var(--light);
      font-weight: 600;
    }

    .table tr {
      transition: var(--transition);
    }

    .table tr:hover {
      background: rgba(67, 97, 238, 0.05);
    }

    .badge {
      padding: 6px 12px;
      border-radius: 20px;
      font-size: 0.85rem;
      font-weight: 600;
    }

    .pending {
      background: rgba(252, 163, 17, 0.2);
      color: #b36a00;
    }

    .confirmed {
      background: rgba(67, 97, 238, 0.2);
      color: var(--primary);
    }

    .delivered {
      background: rgba(76, 201, 240, 0.2);
      color: #0a6c74;
    }

    .cancel-btn {
      background: var(--danger);
      color: white;
      padding: 6px 12px;
      border: none;
      cursor: pointer;
      border-radius: var(--border-radius);
      font-size: 0.85rem;
      transition: var(--transition);
      text-decoration: none;
      display: inline-block;
    }

    .cancel-btn:hover {
      background: #c1121f;
      transform: scale(1.05);
    }

    .footer {
      text-align: center;
      padding: 20px;
      color: var(--gray);
      margin-top: 40px;
    }

    @keyframes slideDown {
      from {
        opacity: 0;
        transform: translateY(-20px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    @keyframes fadeIn {
      from {
        opacity: 0;
        transform: translateY(20px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    @media (max-width: 768px) {
      .header {
        flex-direction: column;
        align-items: flex-start;
      }
      
      .language-switcher {
        margin: 10px 0;
      }
      
      .table {
        display: block;
        overflow-x: auto;
      }
    }
</style>
<script>
  document.addEventListener('DOMContentLoaded', function() {
    // Add animation to table rows
    const rows = document.querySelectorAll('.table tr');
    rows.forEach((row, index) => {
      row.style.animation = `fadeIn 0.5s ease ${index * 0.1}s forwards`;
      row.style.opacity = 0;
    });

    // Add confirmation for cancel action
    const cancelButtons = document.querySelectorAll('.cancel-btn');
    cancelButtons.forEach(button => {
      button.addEventListener('click', function(e) {
        if (!confirm('Are you sure you want to cancel this order?')) {
          e.preventDefault();
        }
      });
    });
  });
</script>
</head>
<body>
<div class="header">
  <div class="brand">
    <div class="logo">PS</div>
    <h1><?php echo t('orders'); ?></h1>
  </div>
  
  <div style="display: flex; align-items: center; gap: 15px;">

    
    <a class="btn secondary" href="user_dashboard.php"><?php echo t('dashboard_link'); ?></a>
    <a class="btn secondary" href="profile.php"><?php echo t('profile_link'); ?></a>
    <a class="btn secondary" href="logout.php"><?php echo t('logout'); ?></a>
  </div>
</div>

<main class="container">
  <?php if($message): ?>
    <div class="new-order-msg"><?php echo $message; ?></div>
  <?php endif; ?>

  <div style="margin-bottom:20px;">
    <a href="user_dashboard.php" class="btn"><?php echo t('place_new_order'); ?></a>
  </div>

  <div class="card">
    <h2><?php echo t('previous_orders'); ?></h2>
    <div class="hr"></div>
    <?php if (empty($orders)): ?>
        <p><?php echo t('no_orders'); ?></p>
    <?php else: ?>
        <table class="table">
            <thead>
                <tr>
                    <th>ID</th>
                    <th><?php echo t('date'); ?></th>
                    <th><?php echo t('area'); ?></th>
                    <th><?php echo t('quantity'); ?></th>
                    <th><?php echo t('status'); ?></th>
                    <th><?php echo t('actions'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($orders as $o): ?>
                <tr>
                    <td><?php echo $o['id']; ?></td>
                    <td><?php echo date('Y-m-d H:i', strtotime($o['created_at'])); ?></td>
                    <td><?php echo htmlspecialchars($o['area_name']); ?></td>
                    <td><?php echo $o['qty']; ?></td>
                    <td>
                        <?php 
                        switch($o['status']){
                            case 'pending': echo '<span class="badge pending">'.t('pending').'</span>'; break;
                            case 'confirmed': echo '<span class="badge confirmed">'.t('confirmed').'</span>'; break;
                            case 'accepted': echo '<span class="badge confirmed">'.t('accepted').'</span>'; break;
                            case 'delivered': echo '<span class="badge delivered">'.t('delivered').'</span>'; break;
                            default: echo '<span class="badge">'.$o['status'].'</span>';
                        }
                        ?>
                    </td>
                    <td>
                        <?php if($o['status']=='pending'): ?>
                            <a href="?cancel_id=<?php echo $o['id']; ?>" class="cancel-btn"><?php echo t('cancel'); ?></a>
                        <?php else: ?>
                            —
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
  </div>
</main>
    <div class="language-switcher">
      <a href="?lang=en" class="lang-btn <?php echo $lang == 'en' ? 'active' : ''; ?>">EN</a>
      <a href="?lang=ar" class="lang-btn <?php echo $lang == 'ar' ? 'active' : ''; ?>">AR</a>
      <a href="?lang=ku" class="lang-btn <?php echo $lang == 'ku' ? 'active' : ''; ?>">KU</a>
    </div>
<div class="footer">© <?php echo date('Y'); ?> <?php echo t('propane_service'); ?></div>
</body>
</html>